<?php
/**
 * Plugin Name: Loopz Gift Cards
 * Plugin URI: https://www.loopz.io/woocommerce
 * Description: Sell and redeem gift cards, refund orders as store credit on your WooCommerce store. Sync gift cards with Clover, Square, Shopify or other WooCommerce stores.
 * Version: 2.0.1
 * Author: Loopz Technologies Inc.
 * License: GPLv3 or later
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 * Developer: Loopz Technologies Inc.
 * Developer URI: https://www.loopz.io
 * Author URI: https://www.loopz.io
 * Requires at least: 6.4
 * Requires PHP:      7.4
 * WC requires at least: 3.2.0
 * WC tested up to: 10.2
 * Requires Plugins: woocommerce
 * WC-HPOS-Compatible: yes
 * Text Domain: loopz-gift-cards
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

define( 'LOOPZ_VERSION', '2.0.0' );
define( 'LOOPZ_SESSION_KEY', 'loopz_gift_card_code' );
define( 'LOOPZ_ORDER_META_CODE_KEY', '_loopz_code' );
define( 'LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY', '_loopz_requested_balance' );
define( 'LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY', '_loopz_redeemed_balance' );

// Prod
define('LOOPZ_API_BASE_URI', 'https://api.loopz.io/v1');
define('LOOPZ_CHECKOUT_HOST', 'https://checkout.loopz.io');

// Dev
//define('LOOPZ_API_BASE_URI', 'https://loopz.ngrok.io/v1');
//define('LOOPZ_CHECKOUT_HOST', 'https://loopz-web.ngrok.io');

define( 'LOOPZ_WC_PLUGIN_DIR_URL', plugin_dir_url( __FILE__ ) );
define( 'LOOPZ_WC_PLUGIN_DIR_PATH', plugin_dir_path( __FILE__ ) );
require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-preferences-class.php';
require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-api-class.php';
require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-leaky-bucket.php';
require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-settings-page.php';
require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-utils.php';

use Loopz\Checkout_Blocks\Extend_Block_Core;
use Loopz\Checkout_Blocks\Blocks_Integration;
use Loopz\Checkout_Blocks\Extend_Store_Endpoint;

add_action( 'init', 'loopz_init' );
add_action( 'plugins_loaded', 'loopz_plugins_loaded' );
add_action( 'before_woocommerce_init', 'loopz_woocommerce_init' );
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'loopz_add_settings_link' );

register_uninstall_hook( __FILE__, 'loopz_uninstall' );
register_activation_hook( __FILE__, 'loopz_activated' );
register_deactivation_hook( __FILE__, 'loopz_deactivated' );

function loopz_plugins_loaded() {
//	load_plugin_textdomain( 'loopz-gift-cards', false, basename( dirname( __FILE__ ) ) . '/languages' );

	// Include other required files
	require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'views/loopz-cart.php';
	require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-cache-class.php';
	require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-diagnostics-class.php';
	require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-woo-hooks.php';
	require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-woo-admin-order-page-hooks.php';
	require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-woo-customer-order-page-hooks.php';
	require_once LOOPZ_WC_PLUGIN_DIR_PATH . 'includes/loopz-woo-gift-product.php';

	// Check if WooCommerce is active and WC_Integration class exists
	if ( class_exists( 'WC_Integration' ) ) {
	} else {
		// WooCommerce is not active
		add_action( 'admin_notices', 'loopz_woocommerce_inactive_notice' );
	}
	$woocommerce_enabled = loopz_preferences::get_woocommerce_enabled();
	if ( $woocommerce_enabled ) {
		add_action( 'woocommerce_blocks_loaded', 'checkout_blocks' );
	}



}

function add_loopz_integration( $integrations ) {
	$integrations[] = 'loopz_settings';

	return $integrations;
}

function checkout_blocks() {
	require_once plugin_dir_path( __FILE__ ) . 'includes/blocks/Blocks_Integration.php';
	require_once plugin_dir_path( __FILE__ ) . 'includes/blocks/Extend_Block_Core.php';
	require_once plugin_dir_path( __FILE__ ) . 'includes/blocks/Extend_Store_Endpoint.php';

	// Initialize classes that depend on WooCommerce
	new Extend_Block_Core();

	Extend_Store_Endpoint::init();

	// Register for Checkout block
	add_action( 'woocommerce_blocks_checkout_block_registration', function ( $integration_registry ) {
		$integration_registry->register( new Blocks_Integration() );
	} );

	// Register for Cart block
	add_action( 'woocommerce_blocks_cart_block_registration', function ( $integration_registry ) {
		$integration_registry->register( new Blocks_Integration() );
	} );

	// Register for mini Cart block
	add_action( 'woocommerce_blocks_mini-cart_block_registration', function ( $integration_registry ) {
		$integration_registry->register( new Blocks_Integration() );
	} );
}

function loopz_init() {
	add_shortcode( 'loopz', 'loopz_shortcode' );

	$current_plugin_version = LOOPZ_VERSION;
	$current_db_version     = loopz_preferences::get_version();
	if ( loopz_preferences::has_api_key()
	     && loopz_preferences::get_woocommerce_enabled() == null
	     && loopz_diagnostics::woocommerce_installed_version() > 0 ) {

		$connected_account = loopz_api::get_woocommerce_connected_account();
		if ( $connected_account != null ) {
			loopz_preferences::set_woocommerce_enabled( true );
		} else {
			loopz_preferences::set_woocommerce_enabled( false );
		}
	}

	loopz_preferences::set_version( $current_plugin_version );
}

function loopz_woocommerce_init() {
    if ( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
	require_once plugin_dir_path( __FILE__ ) . 'includes/loopz-woo-hooks.php';

}

function loopz_add_settings_link( $links ) {
	$settings_link = '<a href="' . admin_url( 'admin.php?page=loopz-gift-cards-settings' ) . '">' . __( 'Settings', 'loopz-gift-cards' ) . '</a>';
	array_unshift( $links, $settings_link );

	return $links;
}

function loopz_activated() {
	if ( loopz_preferences::has_api_key()
	     && loopz_preferences::get_woocommerce_enabled() ) {

		loopz_api::connect_woocommerce_connected_account();
	}
}

function loopz_deactivated() {
    loopz_preferences::disconnect();
}

function loopz_uninstall() {
	remove_shortcode( 'loopz' );
    loopz_api::offboard();
}

/**
 * Register the /auth/completed REST route
 */
add_action( 'rest_api_init', 'loopz_register_auth_completed_route' );

function loopz_register_auth_completed_route() {
	register_rest_route(
		'loopz-gift-cards/v1',
		'/auth/completed',
		array(
			'methods'             => 'PUT',
			'callback'            => 'loopz_handle_auth_completed',
			'permission_callback' => '__return_true',
		)
	);
}

function loopz_handle_auth_completed( WP_REST_Request $request ) {
    $auth_token = loopz_api::get_auth_token();

    if ( empty( $auth_token ) ) {
        return new WP_Error(
            'auth_failed',
            'No valid WooCommerce API token found',
            array( 'status' => 400 )
        );
    }

    $response = wp_remote_get(
        LOOPZ_API_BASE_URI . '/woocommerce/config',
        array(
            'headers' => array(
                'X-Loopz-WooCommerce-Auth-Token' => $auth_token,
                'Content-Type'                  => 'application/json',
            ),
            'timeout' => 45,
        )
    );

    $status_code = wp_remote_retrieve_response_code( $response );
    if ( 200 !== $status_code ) {
        return new WP_Error(
            'auth_failed',
            'Authentication failed',
            array( 'status' => 400 )
        );
    }

    $config = json_decode( wp_remote_retrieve_body( $response ), true );
    if ( ! is_array( $config ) ) {
        return new WP_Error(
            'auth_failed',
            'Unable to parse config',
            array( 'status' => 400 )
        );
    }

    loopz_preferences::set_woocommerce_enabled( true );
    loopz_preferences::set_connection_config( $config );
    loopz_preferences::set_valid_token( $auth_token );

    return new WP_REST_Response( array( 'success' => true ), 200 );
}

/**
 * Register the /config REST route
 */
add_action( 'rest_api_init', 'loopz_register_config_route' );

function loopz_register_config_route() {
    register_rest_route(
        'loopz-gift-cards/v1',
        '/config',
        array(
            'methods'             => 'PUT',
            'callback'            => 'loopz_handle_config',
            'permission_callback' => '__return_true',
        )
    );
}

function loopz_handle_config(WP_REST_Request $request) {
    // Wait a few seconds to ensure the Loopz account is setup and all configuration is available
    sleep(5);
    $api_response = loopz_api::get_config();

   if($api_response->success) {
       $config = $api_response->body;
       loopz_preferences::set_connection_config($config);
   }
}
/**
 * Register the /offboarding REST route. This endpoint is called from Loopz when offboarding a Woo connected account
 */
add_action( 'rest_api_init', 'loopz_register_offboarding_route' );

function loopz_register_offboarding_route() {
    register_rest_route(
        'loopz-gift-cards/v1',
        '/offboarding',
        array(
            'methods'             => 'PUT',
            'callback'            => 'loopz_handle_offboarding',
            'permission_callback' => '__return_true',
        )
    );
}

function loopz_handle_offboarding(WP_REST_Request $request) {
    loopz_preferences::disconnect();
}

/**
 * Register the /connection-status REST route. This endpoint is called from Loopz to check if the plugin's connection status
 */
add_action( 'rest_api_init', 'loopz_register_connection_status_route' );

function loopz_register_connection_status_route() {
    register_rest_route(
        'loopz-gift-cards/v1',
        '/connection-status',
        array(
            'methods'             => 'GET',
            'callback'            => 'loopz_handle_get_connection_status',
            'permission_callback' => '__return_true',
        )
    );
}

function loopz_handle_get_connection_status(WP_REST_Request $request) {
    $connection_status = array(
        'connected' => loopz_preferences::is_connected() ? 'true': 'false',
        'merchantId' => loopz_preferences::get_merchant_id(),
        'programId' => loopz_preferences::get_program_id(),
        'shopDomain' => get_site_url(),
        );

    return new WP_REST_Response($connection_status, 200 );
}

function loopz_woocommerce_inactive_notice() {
	if ( !is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
		$class = 'notice notice-error';
		$message = __( 'Loopz Gift Cards requires WooCommerce to be installed and active.', 'loopz-gift-cards' );
		echo esc_html("<div class='$class'><p>$message</p></div>");
	}
}

// Declare HPOS compatibility for WooCommerce
add_filter( 'woocommerce_feature_enabled', function( $enabled, $feature ) {
    if ( 'custom_order_tables' === $feature ) {
        return true;
    }
    return $enabled;
}, 10, 2 );
