<?php
/**
 * Defines WooCommerce hooks to support applying and redeeming a gift card to an order
 */
if ( loopz_preferences::is_connected() ) {
    if ( loopz_preferences::get_redemption_enabled()) {
        $cart_adjustment_priority = 100;

        // Compatability with WooCommerce Extended Coupon Features PRO
        if ( class_exists( 'WJECF_AutoCoupon' ) ) {
            $cart_adjustment_priority = 20;
        }

        // Compatability with WooCommerce AvaTax
        if ( class_exists( 'WC_AvaTax_Checkout_Handler' ) ) {
            $cart_adjustment_priority = 1000;
        }

        // Compatability with Advanced Dynamic Pricing (AlgolPlus)
        add_filter( 'wdp_calculate_totals_hook_priority', function( $priority ) use ( $cart_adjustment_priority ) { return $cart_adjustment_priority - 1; });


        // Hidden forms
        add_action( 'woocommerce_before_cart', 'loopz_woocommerce_hidden_forms' );
        add_action( 'woocommerce_before_checkout_form', 'loopz_woocommerce_hidden_forms' );

        // Remove the gift card
        add_action( 'woocommerce_cart_emptied', 'loopz_woocommerce_cart_emptied' );

        // Deduct the balance from the gift card
        add_action( 'woocommerce_pre_payment_complete', 'loopz_woocommerce_redeem_gift_card' );
        add_action( 'woocommerce_order_status_processing', 'loopz_woocommerce_redeem_gift_card' );
        add_action( 'woocommerce_order_status_pre-ordered', 'loopz_woocommerce_redeem_gift_card' );
        add_action( 'woocommerce_order_status_completed', 'loopz_woocommerce_redeem_gift_card' );
        add_action( 'woocommerce_payment_complete', 'loopz_woocommerce_redeem_gift_card' );

        // Diagnostics output
        add_action( 'wp_footer', array( 'loopz_diagnostics', 'render') );

	    if ( !is_using_checkout_blocks() ) {
		    // Entering the gift card code on the cart
		    add_action( 'woocommerce_cart_totals_before_order_total', 'loopz_woocommerce_cart_coupon' );

		    // Displaying the gift card code on the checkout/payment page
		    add_action( 'woocommerce_review_order_before_order_total', 'loopz_woocommerce_cart_coupon' );

		    // Adding the gift card code to the order meta data
		    add_action( 'woocommerce_checkout_create_order', 'loopz_woocommerce_checkout_create_order' );

		    // Adjust the cart total to take into consideration the gift card balance
		    add_action( 'woocommerce_after_calculate_totals', 'loopz_woocommerce_after_calculate_totals', $cart_adjustment_priority, 1 );
	    }
    }

    // Rendering the gift card that has been used in an order in the admin order details page
    add_action( 'woocommerce_admin_order_totals_after_tax', 'loopz_woocommerce_admin_order_totals_after_tax' );

	if(!is_using_checkout_blocks()){
		// Rendering the gift card that has been used in an order in the customers view post payment & emails
		add_filter( 'woocommerce_get_order_item_totals', 'loopz_woocommerce_get_order_item_totals', 30, 3 );
	}
}

/*
    Adjust the cart total to take into consideration the gift card balance
    This gets called on the basket page, on the checkoput page and before the order is created
*/
function loopz_woocommerce_after_calculate_totals( $cart ) {
    $debug = loopz_preferences::get_woocommerce_diagnostics_mode();

    loopz_diagnostics::new_group();
    loopz_diagnostics::append( "Entering loopz_woocommerce_after_calculate_totals…" );

    if ( property_exists( $cart, 'recurring_cart_key' ) ) {
        loopz_diagnostics::append( "└ recurring_cart_key property exists, exiting." );
        return;
    }

    if ( $debug ) {
        loopz_diagnostics::append( "├ Incoming cart sub-total: " . strval($cart->get_subtotal()) );
        loopz_diagnostics::append( "├ Incoming cart shipping: " . strval($cart->get_shipping_total()) );
        loopz_diagnostics::append( "├ Incoming cart taxes: " . strval($cart->get_total_tax()) );
        loopz_diagnostics::append( "├ Incoming cart total: " . strval($cart->total) );
    }

    if ( is_cart() || is_checkout() ) {
        if ( isset($_SERVER['REQUEST_METHOD']) && 'POST' == $_SERVER['REQUEST_METHOD'] && isset($_POST['loopz_gift_card_code']) && isset($_POST['loopz_gift_card_nonce'])) {
            $nonce = sanitize_text_field(wp_unslash($_POST['loopz_gift_card_nonce']));

            if(wp_verify_nonce($nonce, 'loopz_gift_card_nonce')) {
                $card_code = sanitize_text_field(wp_unslash($_POST['loopz_gift_card_code']));
                if ( empty($card_code) == false) {
                    $card_code = str_replace(array("-", "_", " "), "", $card_code);
                }

                if (strlen( $card_code ) > 0) {
                    // Check if the cart contains a Loopz managed gift card product and if redemption for gift card orders setting is allowed
                    if(!\loopz_preferences::get_redeem_for_gift_card_order()) {
                        $cart = WC()->cart;
                        $gift_card_product_id = \loopz_preferences::get_product_id();
                        if(isset($cart) && isset($gift_card_product_id)) {
                            $line_items = $cart->get_cart_contents();
                            if(isset($line_items)) {
                                foreach ($line_items as $line_item) {
                                    if($line_item['product_id'] == $gift_card_product_id) {
                                        $card_code = null;
                                    }
                                }
                            }
                        }
                    }

                    loopz_diagnostics::append( "├ Setting gift card code to: " . $card_code );
                } else {
                    loopz_diagnostics::append( "├ Clearing gift card code" );
                }

                loopz_cache::set_gift_card_code( $card_code );
            }
        }
    }

    if ( !loopz_cache::get_gift_card_code() ) {
        loopz_cache::clear();
        loopz_diagnostics::append( "└ No gift card code entered, exiting." );
        return;
    }

    loopz_diagnostics::append( "├ Gift card code: " . loopz_cache::get_gift_card_code() );

    $gift_card_balance = loopz_api::get_gift_card_balance();

    if ( $gift_card_balance <= 0 ) {
        loopz_diagnostics::append( "└ Gift card has no balance, exiting." );
        return;
    }

    // Deal with this hook being replayed in the same request
    if ( property_exists( $cart, 'loopz_new_calculated_total' ) ) {
        if ( $debug ) {
            loopz_diagnostics::append( "├ Replaying compute request…" );
            loopz_diagnostics::append( "├ Previous incoming cart total: " . strval($cart->loopz_previous_incoming_cart_total) );
            loopz_diagnostics::append( "├ Previous applied balance: " . strval($cart->loopz_applied_gift_card_balance) );
            loopz_diagnostics::append( "├ Previous adjusted cart total: " . strval($cart->loopz_new_calculated_total) );
        }

        if ( abs($cart->loopz_new_calculated_total - $cart->total) < 0.00001 ) {
            loopz_diagnostics::append( "└ Cart totals are the same: " . strval($cart->total) . ", exiting." );
            return;
        }

        loopz_diagnostics::append( "└ Cart totals differ by: " . strval($cart->loopz_new_calculated_total - $cart->total) );
    }

    $eligible_cart_total = $cart->total;

    if ( loopz_preferences::get_woocommerce_apply_to_shipping() == false ) {
        $eligible_cart_total = $eligible_cart_total - $cart->get_shipping_total();
        loopz_diagnostics::append( "├ Applying gift card to shipping" );
    }

    if ( loopz_preferences::get_woocommerce_apply_to_taxes() == false ) {
        $eligible_cart_total = $eligible_cart_total - $cart->get_total_tax();
        loopz_diagnostics::append( "├ Applying gift card to taxes" );
    }

    // Allow devs to mutate the cart total to filter items our etc...
    $eligible_cart_total = apply_filters( 'loopz_eligible_cart_total', $eligible_cart_total, $cart );

    // Make sure we don't set the cart total negative
    $applied_gift_card_balance = min($eligible_cart_total , $gift_card_balance);

    loopz_diagnostics::append( "├ Eligible cart total: " . strval($eligible_cart_total) );
    loopz_diagnostics::append( "├ Gift card balance: " . strval($gift_card_balance) );

    if ( $cart->total < $applied_gift_card_balance ) {
        $applied_gift_card_balance = $cart->total;
    }

    $new_cart_total = $cart->total - $applied_gift_card_balance;

    loopz_diagnostics::append( "├ Applied balance: " . strval($applied_gift_card_balance) );
    loopz_diagnostics::append( "└ New cart total: " . strval($new_cart_total) );

    // Store some data about totals for replayability guard check
    $cart->loopz_previous_incoming_cart_total = $cart->total;
    $cart->loopz_new_calculated_total = $new_cart_total;
    $cart->loopz_applied_gift_card_balance = $applied_gift_card_balance;

    $cart->set_total($new_cart_total);

    try {
        $cart->set_session();
    } catch (exception $e) {
        if ( $debug ) {
            loopz_diagnostics::append( "├ Exception calling cart->set_session()" );
            loopz_diagnostics::append( "├ " . $e->getMessage() );
        }
    }

    // Store the gift card applied balance
    loopz_cache::$applied_gift_card_balance = $applied_gift_card_balance;

    loopz_diagnostics::append( "Exiting loopz_woocommerce_after_calculate_totals…" );
}

/*
    Rendering the gift card that has been used in an order in the admin order details page
    This is the admin view when viewing an order. It is not translated.
*/
function loopz_woocommerce_admin_order_totals_after_tax ( $order_id ) {
    $order = wc_get_order( $order_id );

    if ( $order -> meta_exists(LOOPZ_ORDER_META_CODE_KEY) ) {
        $code = trim( $order -> get_meta(LOOPZ_ORDER_META_CODE_KEY) );
        $requested_balance = $order -> get_meta(LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY);
        $outstanding = $requested_balance;

        if ( $order -> meta_exists(LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY) ) {
            $redeemed_balance = $order -> get_meta(LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY);
            $outstanding = $requested_balance - $redeemed_balance;
        }

        if ( $requested_balance > 0 ) {
            ?>
            <tr>
                <td class="label">Gift card (<a href="https://my.loopz.io/card/<?php echo esc_attr($code); // WPCS: XSS ok. ?>" target="_blank" style="text-transform: uppercase"><?php echo esc_attr($code); // WPCS: XSS ok. ?></a>):</td>
                <td width="1%"></td>
                <td class="total">
                    <?php echo wp_kses(wc_price( -1 * $requested_balance, array( 'currency' => $order->get_currency() ) ), ['pre_user_description']); // WPCS: XSS ok. ?>
                </td>
            </tr>
            <?php
            if ( $outstanding > 0 && $order -> get_status() == "on-hold" ) {
                ?>
                <tr>
                    <td colspan="3" style="color: red">
                        Please ensure
                        <?php echo wp_kses(wc_price( $outstanding, array( 'currency' => $order->get_currency() ) ), ['pre_user_description']); // WPCS: XSS ok. ?>
                        has been redeemed from the gift card in Loopz
                    </td>
                </tr>
                <?php
            }
        }
    }
}

/*
    Rendering the gift card that has been used in an order in the customers view post payment & emails
    This is the customers view when viewing an order in My Account and in the emails & receipt
*/
function loopz_woocommerce_get_order_item_totals( $total_rows, $order, $tax_display ) {
    if ( $order -> meta_exists(LOOPZ_ORDER_META_CODE_KEY) ) {
        $code = $order -> get_meta(LOOPZ_ORDER_META_CODE_KEY);
        $applied_balance = $order -> get_meta(LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY);

        if ( $applied_balance > 0 ) {
            // Set last total row in a variable and remove it.
            $grand_total = $total_rows['order_total'];
            unset( $total_rows['order_total'] );

            // Insert a new row
            $total_rows['loopz'] = array(
                'label' => __( 'Gift card', 'loopz-gift-cards' ) . ' (' . strtoupper( $code ) . '):',
                'value' => wc_price( -1 * $applied_balance, array( 'currency' => $order->get_currency() ) ),
            );

            // Set back last total row
            $total_rows['order_total'] = $grand_total;
        }
    }

    return $total_rows;
}

/*
    Store the code & requested balance as soon as we have a new order created
*/
function loopz_woocommerce_checkout_create_order( $order ) {
    $code = loopz_cache::get_gift_card_code();

    if ( isset($code) ) {
        $applied_balance = loopz_cache::$applied_gift_card_balance;

        if ( $applied_balance > 0 ) {
            $order -> add_meta_data( LOOPZ_ORDER_META_CODE_KEY, $code );
            $order -> add_meta_data( LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY, $applied_balance );
            $order -> add_meta_data( LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY, 0 );
        }
    }
}

/*
    Remove the gift card code from the checkout as soon as the order is placed
*/
function loopz_woocommerce_cart_emptied() {
    loopz_cache::clear();
}

/*
    Deduct the balance from the gift card once 'payment has been made' (or for Cash On Delivery type payments, when it's set to 'processing')
    This method is safe to repeatedly call
*/
function loopz_woocommerce_redeem_gift_card( $order_id ) {
	$order = wc_get_order( $order_id );

	if ( $order->meta_exists( LOOPZ_ORDER_META_CODE_KEY ) ) {
		$code              = $order->get_meta( LOOPZ_ORDER_META_CODE_KEY );
		$requested_balance = $order->get_meta( LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY );
		$redeemed_balance  = $order->get_meta( LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY );
		$outstanding       = $requested_balance - $redeemed_balance;

		if ( $outstanding > 0 ) {
			$redeem_response = loopz_api::redeem_gift_card( $code, $outstanding, $order );

			if ( is_array( $redeem_response ) ) {
				$just_redeemed_value = $redeem_response['redeemed_value'];
				$transaction_id      = $redeem_response['transaction_id'];

				// Store the new total redeemed
				$order->update_meta_data(
					LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY,
					floatval( $just_redeemed_value ) + floatval( $redeemed_balance )
				);

				// Store the new gift card transaction ID
				$order->update_meta_data( 'loopz_gift_card_transaction_id', $transaction_id );

				// Add an order note
				$order->add_order_note(
					wc_price( $just_redeemed_value, array( 'currency' => $order->get_currency() ) )
					. " redeemed from gift card " . strtoupper( $code )
					. ". Transaction ID: " . $transaction_id
				);

				$order->save();
			}

			loopz_woocommerce_possibly_hold_order( $order_id );
		}
	}
}

/*
    If we've had an exceptional reason to put the order on hold
    This only really occurs when there either:
    1) Not enough balance on the gift card after all (i.e. bad actor)
    2) There was a transient API call failure
*/
function loopz_woocommerce_possibly_hold_order( $order_id ) {
    $order = wc_get_order( $order_id );

    if ( $order -> meta_exists(LOOPZ_ORDER_META_CODE_KEY) ) {
        $code = $order -> get_meta(LOOPZ_ORDER_META_CODE_KEY);
        $requested_balance = $order -> get_meta(LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY);
        $redeemed_balance = $order -> get_meta(LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY);
        $outstanding = $requested_balance - $redeemed_balance;

        if ( $outstanding > 0 ) {
            if ( $redeemed_balance > 0 ) {
                $order -> add_order_note( "Gift card \"" . strtoupper( $code ) . "\" was not redeemed correctly. Only " . wc_price( $redeemed_balance, array( 'currency' => $order->get_currency() ) ) . " was redeemed off the gift card. Please redeem a further " . wc_price( $outstanding, array( 'currency' => $order->get_currency() ) ) . " from the gift card in Loopz" );
                $order -> save();
            }
            else {
                $order -> add_order_note( "Gift card \"" . strtoupper( $code ) . "\" was not redeemed as the gift card does not have enough remaining balance at the point of payment." );
                $order -> save();
            }

            $status = $order -> get_status();

            if ( $status != 'on-hold' && $status != 'completed' ) {
                $order -> set_status('on-hold');
                $order -> save();
            }
        }
    }
}


/*
 * Check if we are currently using the new woocommerce checkout blocks
 */
function is_using_checkout_blocks() {
    $checkout_page_id = null;
    if ( function_exists( 'wc_get_page_id' ) ) {
        $checkout_page_id = wc_get_page_id( 'checkout' );
    } else {
        $checkout_page_id = -1;
    }

	$has_block_checkout = $checkout_page_id && has_block( 'woocommerce/checkout', $checkout_page_id );

	if ( $has_block_checkout) {
		$is_using_checkout_block=true;
	}
	else{
		$is_using_checkout_block=false;
	}
	return $is_using_checkout_block;
}


