<?php
/**
 * Class Blocks/Extend_Block_Core file.
 *
 * @package Loopz\Blocks
 */

namespace Loopz\Checkout_Blocks;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Extend_Block_Core {

	/**
	 * Plugin Identifier, unique to each plugin.
	 *
	 * @var string
	 */
	private $name = 'loopz';

	/**
	 * Bootstraps the class and hooks required data.
	 */
	public function __construct() {

		// Initialize hooks
		add_action( 'woocommerce_store_api_checkout_update_order_from_request', array(
			$this,
			'save_loopz_checkout_code'
		), 10, 2 );

		// Register the update callback when WooCommerce Blocks is loaded
		add_action( 'init', array( $this, 'register_store_api_callback' ) );

		// Register fee calculation
		add_action( 'woocommerce_cart_calculate_fees', array( $this, 'loopz_add_gift_card_discount' ), 20, 1 );

        // Register a filter to determine if the Stripe express checkout should be hidden if a gift card is applied
        add_filter('wc_stripe_should_hide_express_checkout_button_based_on_tax_setup', array( $this, 'loopz_hide_stripe_express_checkout' ), 10, 3);

		// Register AJAX actions
		$this->register_ajax_actions();
	}

	/**
	 * Registers AJAX actions.
	 */
	private function register_ajax_actions() {
		add_action( 'wp_ajax_loopz_apply_gift_card', array( $this, 'handle_apply_gift_card' ) );
		add_action( 'wp_ajax_nopriv_loopz_apply_gift_card', array( $this, 'handle_apply_gift_card' ) );
		// Removed handle_remove_gift_card actions as removal is handled via Store API
	}

	/**
	 * Register the update callback with WooCommerce Store API
	 */
	public function register_store_api_callback() {
		if ( function_exists( 'woocommerce_store_api_register_update_callback' ) ) {
			woocommerce_store_api_register_update_callback(
				array(
					'namespace' => $this->name,
					'callback'  => array( $this, 'loopz_store_api_callback' ),
				)
			);
		}
	}

	/**
	 * Callback function for the Store API update
	 *
	 * @param array $data Data sent from the client.
	 */
	public function loopz_store_api_callback( $data ) {
		if ( isset( $data['action'] ) ) {
			if ( 'update_gift_card_discount' === $data['action'] ) {
				$price = isset( $data['price'] ) ? floatval( $data['price'] ) : 0;
				$code  = isset( $data['code'] ) ? sanitize_text_field( $data['code'] ) : '';

				if ( $price > 0 && ! empty( $code ) ) {
					// Store the gift card amount and code in session
					WC()->session->set( LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY, $price );
					WC()->session->set( LOOPZ_SESSION_KEY, $code );
				}
			} elseif ( 'remove_gift_card_discount' === $data['action'] ) {
				// Remove the gift card from the session
				WC()->session->__unset( LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY );
				WC()->session->__unset( LOOPZ_SESSION_KEY );
			}
		}
	}

	/**
	 * Handles applying the gift card via AJAX.
	 */
	public function handle_apply_gift_card() {
		// Verify nonce
        if(!isset($_POST['nonce'])) {
            wp_send_json_error( array( 'message' => 'Invalid nonce.' ), 400 );
            wp_die();
        }
        $nonce = sanitize_text_field(wp_unslash($_POST['nonce']));
		if ( ! isset( $nonce ) || !wp_verify_nonce($nonce, 'loopz_gift_card_nonce' ) ) {
			wp_send_json_error( array( 'message' => 'Gift card not found.' ), 400 );
			wp_die();
		}

		// Check if gift_card_code is provided
		if ( isset( $_POST['gift_card_code'] ) ) {
			$gift_card_code = sanitize_text_field( wp_unslash($_POST['gift_card_code']) );

			/*
			 * Check if leaky-bucket rate limit here
			 */
			$limiter = new \Loopz_Leaky_Bucket( 5, 60 );

			if ( ! $limiter->is_allowed() ) {
				// Bucket is full.
				wp_send_json_error(
					array( 'message' => __( 'Gift card cannot be applied. Please try again later.', 'loopz-gift-cards' ) ),
					429
				);
				wp_die();
			}

			// If allowed, increment usage before validating the code.
			$limiter->increment();

            // Check if the cart contains a Loopz managed gift card product and if redemption for gift card orders setting is allowed
            if(!\loopz_preferences::get_redeem_for_gift_card_order()) {
                $cart = WC()->cart;
                $gift_card_product_id = \loopz_preferences::get_product_id();
                if(isset($cart) && isset($gift_card_product_id)) {
                    $line_items = $cart->get_cart_contents();
                    if(isset($line_items)) {
                        foreach ($line_items as $line_item) {
                            if($line_item['product_id'] == $gift_card_product_id) {
                                wp_send_json_error( array( 'message' =>  __( 'A gift card cannot be redeemed for a gift card order.', 'loopz-gift-cards' )  ), 400 );
                                wp_die();
                            }
                        }
                    }
                }
            }

			// Fetch the gift card using get_gift_card
			$loopz_api    = new \loopz_api();
			$gift_card    = $loopz_api::get_gift_card( $gift_card_code );
			$card_balance = $loopz_api::get_gift_card_balance( $gift_card_code );

			if ( is_null( $gift_card ) ) {
				// Gift card not found
				wp_send_json_error( array( 'message' => __('Gift card not found.', 'loopz-gift-cards') ), 400 );
				wp_die();
			}


			if ( $card_balance > 0 ) {

				// Store the gift card amount and code in session
				WC()->session->set( LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY, $card_balance );
				WC()->session->set( LOOPZ_SESSION_KEY, $gift_card_code );

				wp_send_json_success(
					array(
						'gift_card_code'     => $gift_card_code,
						'gift_card_value'    => $card_balance,
						'gift_card_currency' => get_woocommerce_currency_symbol(),
					),
					200
				);
			} elseif ( $card_balance == 0 ) {

				// Balance is 0: No remaining balance
				wp_send_json_error( array( 'message' => 'Gift card has no remaining balance.' ), 400 );
				wp_die();
			} else {

				// Any other error: Gift card not found
				wp_send_json_error( array( 'message' => 'Gift card not found.' ), 400 );
				wp_die();
			}
		} else {

			// Fetch the applied gift card
			$gift_card_code   = WC()->session->get( LOOPZ_SESSION_KEY, '' );
			$gift_card_amount = WC()->session->get( LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY, 0 );

			if ( ! empty( $gift_card_code ) ) {
				wp_send_json_success(
					array(
						'gift_card_code'     => $gift_card_code,
						'gift_card_value'    => $gift_card_amount,
						'gift_card_currency' => get_woocommerce_currency_symbol(),
					),
					200
				);
			} else {
				wp_send_json_success( array(), 200 );
			}
		}

		wp_die();
	}


	/**
	 * Adds the gift card discount to the WooCommerce cart.
	 *
	 * @param \WC_Cart $cart The WooCommerce cart object.
	 */
	public function loopz_add_gift_card_discount( $cart ) {
		if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
			return;
		}

		// Get the gift card amount and code from session
		$discount_amount = WC()->session->get( LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY, 0 );
		$gift_card_code  = WC()->session->get( LOOPZ_SESSION_KEY, '' );

		// Remove existing Loopz gift card fees if they exist
		$new_fees = array();
		foreach ( $cart->get_fees() as $fee ) {
			if ( strpos( $fee->name, __( 'Gift card', 'loopz-gift-cards' ) ) === false ) {
				$new_fees[] = $fee;
			}
		}
		$cart->fees_api()->set_fees( $new_fees );

		if ( $discount_amount > 0 ) {
			// Calculate eligible cart total
			$eligible_cart_total = $cart->get_subtotal();

            $apply_to_shipping = \loopz_preferences::get_woocommerce_apply_to_shipping();
			if ( $apply_to_shipping ) {
				$eligible_cart_total += $cart->get_shipping_total();
				\loopz_diagnostics::append( "├ Applying gift card to shipping" );
			}

            $apply_to_taxes = \loopz_preferences::get_woocommerce_apply_to_taxes();
			if ( $apply_to_taxes ) {
				$eligible_cart_total += $cart->get_total_tax();
				\loopz_diagnostics::append( "├ Applying gift card to taxes" );
			}

			// Allow developers to modify eligible cart total
			$eligible_cart_total = apply_filters( 'loopz_eligible_cart_total', $eligible_cart_total, $cart );

			// Ensure the discount does not exceed the eligible cart total
			$applied_discount = min( $discount_amount, $eligible_cart_total );

			// Ensure the discount does not exceed the cart total
			$applied_discount = min( $applied_discount, $cart->get_total() );

			// Add the discount as a negative fee
			$fee_label = __( 'Gift card', 'loopz-gift-cards' );
			if ( ! is_numeric( $applied_discount ) ) {
				$applied_discount = 0; //
			}
			WC()->session->set( LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY, (float) $applied_discount );

			$cart->add_fee( $fee_label, - (float) $applied_discount, false, '' );
		}
	}



    /**
     * Check if the Stripe express checkout should be hidden if a gift card is applied
     *
     * @param $hide_based_on_tax
     * @return boolean
     */
    function loopz_hide_stripe_express_checkout($hide_based_on_tax) {
        $cart = WC()->cart;

        if (isset($cart)) {
            $fees = $cart->get_fees();

            foreach ( $fees as $fee ) {
                if($fee->name == __('Gift card', 'loopz-gift-cards')) {
                    $hide_based_on_tax = true;
                }
            }
        }

        return $hide_based_on_tax;
    }


	/**
	 * Saves the loopz gift card to the order's metadata.
	 *
	 * @param \WC_Order $order Order object.
	 * @param \WP_REST_Request $request REST request object.
	 */
	public function save_loopz_checkout_code( \WC_Order $order, \WP_REST_Request $request ) {

		$gift_card_code = WC()->session->get( LOOPZ_SESSION_KEY, '' );

		if ( isset( $gift_card_code ) ) {
			$discount_amount = WC()->session->get( LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY, 0 );

			if ( $discount_amount > 0 ) {
				$order->update_meta_data( LOOPZ_ORDER_META_CODE_KEY, $gift_card_code );
				$order->update_meta_data( LOOPZ_ORDER_META_REQUESTED_BALANCE_KEY, $discount_amount );
				$order->update_meta_data( LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY, 0 );
			}
		}


		/**
		 * Save the order to persist changes
		 */
		$order->save();
	}

}
