import {useEffect, useState} from '@wordpress/element';
import {__, sprintf} from '@wordpress/i18n';
import {Spinner} from '@wordpress/components';
import {Button, Panel, RemovableChip, ValidatedTextInput, ValidationInputError} from '@woocommerce/blocks-components';
import {useDispatch, useSelect} from '@wordpress/data';
import {getSetting} from '@woocommerce/settings';

const localizedStrings = loopzGiftCardData.localizedStrings;

/**
 * The LoopzGiftCardBlock component allows users to apply a gift card code.
 */
export const LoopzGiftCardBlock = () => {
    const [isExpanded, setIsExpanded] = useState(false);
    const [giftCardCode, setGiftCardCode] = useState('');
    const [giftCardCurrency, setGiftCardCurrency] = useState('');
    const [isApplying, setIsApplying] = useState(false);
    const [giftCardApplied, setGiftCardApplied] = useState(false);
    const [giftCardValue, setGiftCardValue] = useState(0);
    const [isLoading, setIsLoading] = useState(false);
    const storeCurrency = getSetting('currency', {});
    const currencySymbol = storeCurrency.symbol; // e.g., "$"

    // Access global data passed from PHP
    const loopzGiftCardData = window.loopzGiftCardData || {};

    // Define the validation store key
    const VALIDATION_STORE_KEY = 'wc/store/validation';

    // Get dispatch functions for the validation store
    const {setValidationErrors, clearValidationErrors} = useDispatch(VALIDATION_STORE_KEY);

    // Get validation errors from the validation store
    const validationError = useSelect((select) => {
        const store = select(VALIDATION_STORE_KEY);
        return store.getValidationError('loopz-gift-card-error'); // Unique error ID
    }, []);

    /**
     * Fetch applied gift card info on component mount.
     */
    useEffect(() => {
        setIsLoading(true);

        const formData = new URLSearchParams();
        formData.append('action', 'loopz_apply_gift_card'); // Using the same AJAX action to fetch applied gift card
        formData.append('nonce', loopzGiftCardData.nonce);
        // No gift_card_code parameter here

        fetch(loopzGiftCardData.ajax_url, {
            method: 'POST',
            body: formData,
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            timeout: 5000
        })
            .then(response => response.json())
            .then(responseBody => {
                if (responseBody.success && responseBody.data.gift_card_code) {
                    setGiftCardApplied(true);
                    setGiftCardCode(responseBody.data.gift_card_code);
                    setGiftCardValue(responseBody.data.gift_card_value);
                    setGiftCardCurrency(responseBody.data.gift_card_currency);
                    // Clear any existing validation errors
                    clearValidationErrors(['loopz-gift-card-error']);
                }
            })
            .catch(errorResponse => {
                setValidationErrors({
                    'loopz-gift-card-error': {
                        message: localizedStrings.gift_card_not_found,
                        hidden: false,
                    },
                });
            }).finally(() => {
            setIsLoading(false);
        });
    }, [clearValidationErrors, loopzGiftCardData.nonce, loopzGiftCardData.ajax_url]);

    /**
     * Handle applying the gift card.
     */
    const onApplyGiftCard = async (event) => {
        event.preventDefault();

        // Clear previous validation errors
        clearValidationErrors(['loopz-gift-card-error']);

        // Validate input
        if (!giftCardCode.trim()) {
            setValidationErrors({
                'loopz-gift-card-error': {
                    message: localizedStrings.please_enter_gift_card_code,
                    hidden: false,
                },
            });
            return;
        }

        setIsApplying(true);

        const formData = new URLSearchParams();
        formData.append('action', 'loopz_apply_gift_card');
        formData.append('nonce', loopzGiftCardData.nonce);
        formData.append('gift_card_code', giftCardCode.trim());

        fetch(loopzGiftCardData.ajax_url, {
                method: 'POST',
                body: formData,
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                }, timeout: 10000
            }
        ).then(response => response.json())
            .then(responseBody => {
                if(responseBody.success) {
                    const data = responseBody.data;
                    setGiftCardApplied(true);
                    setGiftCardValue(data.gift_card_value);
                    setGiftCardCurrency(data.gift_card_currency);
                    setIsExpanded(false);

                    // Clear validation errors on success
                    clearValidationErrors(['loopz-gift-card-error']);

                    // Update the cart total using the Store API
                    try {
                        const {extensionCartUpdate} = window.wc.blocksCheckout || {};

                        if (typeof extensionCartUpdate === 'function') {
                            extensionCartUpdate({
                                namespace: 'loopz',
                                data: {
                                    action: 'update_gift_card_discount',
                                    price: parseFloat(data.gift_card_value),
                                    code: data.gift_card_code,
                                },
                            });
                        }
                    } catch {
                        // If updating the cart fails, default to "Gift card not found"
                        setValidationErrors({
                            'loopz-gift-card-error': {
                                message: localizedStrings.gift_card_not_found,
                                hidden: false,
                            },
                        });
                    }
                } else if(!responseBody.success) {
                    const backendErrorMessage = responseBody.data?.message || localizedStrings.gift_card_not_found;
                    setValidationErrors({
                        'loopz-gift-card-error': {
                            message: __(backendErrorMessage, 'loopz-gift-cards'),
                            hidden: false,
                        },
                    });
                }

            })
            .catch(errorResponse => {
                const backendErrorMessage = errorResponse.response?.data?.data?.message || localizedStrings.gift_card_not_found;
                setValidationErrors({
                    'loopz-gift-card-error': {
                        message: __(backendErrorMessage, 'loopz-gift-cards'),
                        hidden: false,
                    },
                });
            })
            .finally(() => {
                setIsApplying(false);
            });
    };

    /**
     * Handle removing the gift card.
     */
    const onRemoveGiftCard = async () => {
        setIsLoading(true);

        // Clear previous validation errors
        clearValidationErrors(['loopz-gift-card-error']);

        try {
            // Update the cart to remove the gift card discount
            const {extensionCartUpdate} = window.wc.blocksCheckout || {};

            if (typeof extensionCartUpdate === 'function') {
                await extensionCartUpdate({
                    namespace: 'loopz',
                    data: {
                        action: 'remove_gift_card_discount',
                    },
                });
            }

            setGiftCardApplied(false);
            setGiftCardValue(0);
            setGiftCardCode('');
            setIsExpanded(false);
        } catch {
            // If removing fails, default to "Gift card not found"
            setValidationErrors({
                'loopz-gift-card-error': {
                    message: localizedStrings.gift_card_not_found,
                    hidden: false,
                },
            });
        } finally {
            setIsLoading(false);
        }
    };


    function handleGiftCodeInputChange(value) {
        setGiftCardCode(value);
        clearValidationErrors(['loopz-gift-card-error']);
    }

    function getAppliedGiftCardLabel() {
        return '•••• ' + giftCardCode.substring(giftCardCode.length - 4);
    }

    if (!loopzGiftCardData.redemptionEnabled) {
        return;
    }

    if (isLoading) {
        return <Spinner/>;
    }


    return (
        <div
            className="wp-block-woocommerce-checkout-order-summary-coupon-form-block wc-block-components-totals-wrapper">
            {giftCardApplied && (
                <Panel
                    title={localizedStrings.applied_gift_card}
                    className="wc-block-components-totals-coupon wc-block-components-panel"
                    initialOpen={true}
                >
                    <RemovableChip
                        key={'gift-card-' + giftCardCode}
                        text={getAppliedGiftCardLabel()}
                        screenReaderText={sprintf(
                            localizedStrings.gift_card_with_code,
                            giftCardCode
                        )}
                        disabled={isLoading}
                        onRemove={onRemoveGiftCard}
                        radius="large"
                        ariaLabel={sprintf(
                            localizedStrings.gift_card_quoted,
                            giftCardCode
                        )}
                    />
                </Panel>
            )}
            {!giftCardApplied && (
                <Panel
                    title={localizedStrings.add_a_gift_card}
                    className="wc-block-components-totals-coupon wc-block-components-panel"
                >
                    <form
                        onSubmit={onApplyGiftCard}
                        className="wc-block-components-totals-coupon__form"
                    >
                        <ValidatedTextInput
                            id={'loopz-gift-card-input'}
                            type="text"
                            label={localizedStrings.enter_gift_code}
                            ariaLabel={localizedStrings.enter_gift_code}
                            value={giftCardCode}
                            onChange={handleGiftCodeInputChange}
                            focusOnMount
                            className="wc-block-components-text-input wc-block-components-totals-coupon__input"
                        />

                        <Button

                            type={'submit'}
                            variant={'contained'}
                            disabled={isApplying || !giftCardCode.trim()}
                            className={'wc-block-components-button wp-element-button wc-block-components-totals-coupon__button contained'}

                        >
                            {localizedStrings.apply}
                        </Button>
                    </form>
                    <ValidationInputError
                        errorMessage={validationError && validationError?.message ? validationError.message : null}/>
                </Panel>
            )}
        </div>
    );
};
