import React, {useEffect, useState} from 'react';
import {createRoot} from 'react-dom/client';
import {
    __experimentalHeading as Heading,
    __experimentalText as Text,
    __experimentalVStack as VStack,
    __experimentalHStack as HStack,
    Button,
    Card,
    CardBody,
    CardFooter,
    CardHeader,
    CheckboxControl,
    DropdownMenu,
    Notice,
    Path, Spinner,
    SVG,
    TabPanel,
    ToggleControl
} from '@wordpress/components';
import {__} from '@wordpress/i18n';
import Badge from "./components/Badge";

const initialRedemptionEnabled = typeof window.loopzSettings !== 'undefined' ? !!window.loopzSettings.redemptionEnabled : false;
const initialApplyToShipping = typeof window.loopzSettings !== 'undefined' ? !!window.loopzSettings.applyToShipping : false;
const initialApplyToTaxes = typeof window.loopzSettings !== 'undefined' ? !!window.loopzSettings.applyToTaxes : false;
const initialRedeemForGiftCardOrder = typeof window.loopzSettings !== 'undefined' ? !!window.loopzSettings.redeemForGiftCardOrder : false;
const initialConnected = typeof window.loopzSettings !== 'undefined' ? !!window.loopzSettings.connected : false;
const initialDashboardLoginUrl = typeof window.loopzSettings !== 'undefined' ? window.loopzSettings.dashboardLoginUrl : null;
const initialProductPermalink = typeof window.loopzSettings !== 'undefined' ? window.loopzSettings.productPermalink : null;
const nonce = typeof window.loopzSettings !== 'undefined' ? window.loopzSettings.nonce : null;
const localizedStrings = window.loopzSettings.localizedStrings;
const dateFormatter = new Intl.DateTimeFormat(loopzSettings.locale,  {day: '2-digit', month: 'short', year: 'numeric'});

const App = () => {
    const [redemptionEnabled, setRedemptionEnabled] = useState(initialRedemptionEnabled);
    const [applyToShipping, setApplyToShipping] = useState(initialApplyToShipping);
    const [applyToTaxes, setApplyToTaxes] = useState(initialApplyToTaxes);
    const [redeemForGiftCardOrder, setRedeemForGiftCardOrder] = useState(initialRedeemForGiftCardOrder);
    const [notice, setNotice] = useState(null);
    const [saving, setSaving] = useState(false);
    const [connected, setConnected] = useState(initialConnected);
    const [disconnecting, setDisconnecting] = useState(false);
    const [dashboardLoginUrl, setDashboardLoginUrl] = useState(initialDashboardLoginUrl);
    const [productPermalink] = useState(initialProductPermalink);
    const [loadingGiftCards, setLoadingGiftCards] = useState(false);
    const [giftCards, setGiftCards] = useState([]);
    const [giftCardsCursor, setGiftCardsCursor] = useState(null);
    const [apiError, setApiError] = useState(null);

    // Check for ?dashboard-login=true in the URL and trigger backend redirect if present
    const params = new URLSearchParams(window.location.search);
    const isDashboardLogin = params.get('dashboard-login') === 'true';
    const initialTabName = params.get('tab') || 'gift-cards';

    useEffect(() => {
        if (initialDashboardLoginUrl && isDashboardLogin) {
            window.location.href = initialDashboardLoginUrl;
        }
    }, []);

    if (connected && !isDashboardLogin) {
        useEffect(() => {
            // Fetch additional account settings on mount
            const fetchAdditionalSettings = async () => {
                try {
                    const formData = new FormData();
                    formData.append('action', 'loopz_get_additional_account_settings');
                    const response = await fetch(ajaxurl, {
                        method: 'POST',
                        credentials: 'same-origin',
                        body: formData,
                    });
                    const result = await response.json();
                    console.log('Additional account settings:', result);
                } catch (e) {
                    console.error('Failed to fetch additional account settings', e);
                }
            };
            fetchAdditionalSettings();

            getGiftCards();
        }, []);
    }

    const handleSave = async () => {
        setSaving(true);
        setNotice(null);
        try {
            const formData = new FormData();
            formData.append('action', 'loopz_save_settings');
            formData.append('applyToShipping', applyToShipping);
            formData.append('applyToTaxes', applyToTaxes);
            formData.append('redeemForGiftCardOrder', redeemForGiftCardOrder);
            formData.append('redemptionEnabled', redemptionEnabled);
            formData.append('nonce', nonce);
            const response = await fetch(ajaxurl, {
                method: 'POST',
                credentials: 'same-origin',
                body: formData,
            });
            const data = await response.json();
            if (data.success) {
                setNotice({status: 'success', message: localizedStrings.settings_saved});
            } else {
                setNotice({
                    status: 'error',
                    message: data.data && data.data.message ? data.data.message : localizedStrings.error_saving_settings
                });
            }
        } catch (e) {
            setNotice({status: 'error', message: localizedStrings.error_saving_settings});
        } finally {
            setSaving(false);
            setTimeout(() => setNotice(null), 3000);
        }
    };

    const handleConnect = async () => {
        setSaving(true);
        setNotice(null);
        try {
            const formData = new FormData();
            formData.append('action', 'loopz_connect_account');
            const response = await fetch(ajaxurl, {
                method: 'POST',
                credentials: 'same-origin',
                body: formData,
            });
            const result = await response.json();
            if (result.success && result.data) {
                if (result.data.url) {
                    window.location.href = result.data.url;
                } else if (result.data.connected) {
                    window.location.reload();
                }
            } else {
                setNotice({
                    status: 'error',
                    message: result.data && result.data.message ? result.data.message : localizedStrings.error_connecting_to_loopz
                });
            }
        } catch (e) {
            setNotice({status: 'error', message: localizedStrings.error_connecting_to_loopz});
        } finally {
            setSaving(false);
            setTimeout(() => setNotice(null), 3000);
        }
    };

    const handleDisconnect = async () => {
        if (!window.confirm(localizedStrings.disconnect_confirmation)) {
            return;
        }
        setDisconnecting(true);
        setNotice(null);
        try {
            const formData = new FormData();
            formData.append('action', 'loopz_disconnect_account');
            const response = await fetch(ajaxurl, {
                method: 'POST',
                credentials: 'same-origin',
                body: formData,
            });
            const data = await response.json();
            if (data.success) {
                setConnected(false);
                setNotice({status: 'success', message: localizedStrings.disconnected_from_loopz});
            } else {
                setNotice({
                    status: 'error',
                    message: data.data && data.data.message ? data.data.message : localizedStrings.error_disconnecting_from_loopz
                });
            }
        } catch (e) {
            setNotice({status: 'error', message: localizedStrings.error_disconnecting_from_loopz});
        } finally {
            setDisconnecting(false);
            document.location.reload();
        }
    };

    if (isDashboardLogin) {
        return (
            <div style={{maxWidth: 600, margin: '2rem auto'}}>
                <Card>
                    <CardBody>
                        <div style={{
                            width: '100%',
                            display: 'flex',
                            flexDirection: 'column',
                            alignItems: 'center',
                            justifyContent: 'center',
                            textAlign: 'center'
                        }}>
                            <img
                                src={window.loopzSettings?.logoUrl || '/wp-content/plugins/loopz-gift-cards/views/images/logo.png'}
                                alt="Loopz Logo"
                                style={{height: 80, marginBottom: 24}}
                            />
                            <div style={{fontSize: '1.1em', color: '#333', marginBottom: 24}}>
                                {localizedStrings.logging_into_dashboard}
                            </div>
                            <Button
                                variant='primary'
                                href={dashboardLoginUrl}
                                target="_blank"
                            >
                                {localizedStrings.open_dashboard}
                            </Button>
                        </div>
                    </CardBody>
                </Card>
            </div>
        )
    }

    async function getGiftCards() {
        setLoadingGiftCards(true);
        try {

            const formData = new FormData();
            formData.append('action', 'loopz_list_gift_cards');
            const response = await fetch(ajaxurl, {
                method: 'POST',
                credentials: 'same-origin',
                body: formData,
            })
            const responseBody = await response.json();
            if(responseBody.success) {
                if(responseBody.data.success) {
                    setGiftCards(responseBody.data.body.giftCards);
                    setGiftCardsCursor(responseBody.data.body.cursor);
                } else {
                    setApiError({
                        message: responseBody.data.body.message ? responseBody.data.body.message : localizedStrings.error_retrieving_gift_cards,
                    });
                }
            } else {
                setApiError({
                    message: localizedStrings.error_retrieving_gift_cards
                });
            }
            setGiftCards(responseBody.data.body.giftCards);
            setLoadingGiftCards(false);
        } catch (e) {
            setLoadingGiftCards(false);
        }
    }

    function renderSettingsTabPanel() {
        return (
            <div style={{maxWidth: '600px', margin: '2rem auto'}}>
                <Card>
                    <CardHeader>
                        <Heading level={4}>
                            {localizedStrings.account_status}
                            <Badge intent="success" style={{marginLeft: '1rem', marginRight: '1rem'}}>
                                {localizedStrings.connected}
                            </Badge>
                        </Heading>
                        <div style={{display: 'flex', gap: '1rem', alignItems: 'center'}}>
                            <Button
                                variant='primary'
                                href={dashboardLoginUrl}
                                target="_blank"
                            >
                                {localizedStrings.open_dashboard}
                            </Button>
                            <DropdownMenu
                                controls={[
                                    {
                                        onClick: handleDisconnect,
                                        title: 'Disconnect'

                                    }
                                ]}
                                icon={<SVG viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><Path
                                    d="M12 8c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2zm0 2c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2zm0 6c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2z"/></SVG>}
                                onToggle={() => {
                                }}
                            />
                        </div>
                    </CardHeader>
                </Card>
                <div style={{marginTop: '1em'}}>
                    <Card>
                        <CardHeader>
                            <VStack>
                                <Heading level={4}>
                                    {localizedStrings.gift_card_redemption_settings}
                                </Heading>
                                <Text variant={'muted'}>
                                    {localizedStrings.control_gift_card_balance}
                                </Text>
                            </VStack>
                        </CardHeader>
                        <CardBody>
                            <ToggleControl
                                label={
                                    <>
                                        {localizedStrings.gift_card_redemption}
                                        <Badge intent={redemptionEnabled ? 'success' : 'error'} style={{marginLeft: '1rem'}}>
                                            {redemptionEnabled ? localizedStrings.enabled : localizedStrings.disabled}
                                        </Badge>
                                    </>
                                }
                                help={localizedStrings.enables_gift_card_redemption}
                                checked={redemptionEnabled}
                                onChange={setRedemptionEnabled}
                            />
                            <CheckboxControl
                                label={localizedStrings.apply_to_shipping}
                                help={localizedStrings.allow_shipping_application}
                                checked={applyToShipping}
                                onChange={setApplyToShipping}
                            />
                            <CheckboxControl
                                label={localizedStrings.apply_to_taxes}
                                help={localizedStrings.allow_tax_application}
                                checked={applyToTaxes}
                                onChange={setApplyToTaxes}
                            />
                            <CheckboxControl
                                label={localizedStrings.redeem_for_gift_card_orders}
                                help={localizedStrings.allow_gift_card_redemption}
                                checked={redeemForGiftCardOrder}
                                onChange={setRedeemForGiftCardOrder}
                            />
                        </CardBody>
                        <CardFooter>
                            <div style={{width: '100%', display: 'flex', justifyContent: 'flex-end'}}>
                                <Button
                                    variant='primary'
                                    onClick={handleSave}
                                    isBusy={saving}
                                    disabled={saving}
                                >{localizedStrings.save_changes}</Button>
                            </div>
                        </CardFooter>
                    </Card>
                </div>
            </div>
        )
    }

    function resolveGiftCardStatusBadgeIntent(status) {
        switch(status) {
            case 'ACTIVE':
                return 'success';
            case 'SUSPENDED':
            case 'INACTIVE':
                return 'warning';
            case 'CLOSED':
                return 'error';
            default:
                return 'default';
        }
    }

    function formatCardNumber(cardNumber) {
        if(cardNumber.length === 16) {
            return cardNumber.substring(0,4) + '-' + cardNumber.substring(4,8) + '-' + cardNumber.substring(8,12) + '-' + cardNumber.substring(12, 16);
        }
        return cardNumber;
    }

    function renderGiftCardsTabPanel() {
        return (
            <div style={{maxWidth: '800px', margin: '2rem auto'}}>
                <Card>
                    <CardHeader>
                        <VStack>
                            <Heading level={4}>
                                {localizedStrings.gift_cards}
                            </Heading>
                        </VStack>
                        <div style={{display: 'flex', gap: '1rem', alignItems: 'center'}}>
                            <Button
                                variant='primary'
                                href={dashboardLoginUrl + '&redirect=' + encodeURIComponent('/gift-cards')}
                                target="_blank"
                            >
                                {localizedStrings.view_all_gift_cards}
                            </Button>
                        </div>
                    </CardHeader>
                    <CardBody style={{padding: 0}}>
                        {loadingGiftCards && (
                            <HStack justify={'center'} style={{padding: '16px'}}>
                                <Spinner style={{
                                    height: 'calc(4px * 10)',
                                    width: 'calc(4px * 10)'
                                }}/>
                            </HStack>

                        )}
                        {!loadingGiftCards && apiError && (
                            <HStack justify={'center'} style={{padding: '16px'}}>
                                <Notice status="error">{apiError.message}</Notice>
                            </HStack>
                        )}
                        {!loadingGiftCards && giftCards && (
                            <table className="loopz-gift-cards-table">
                                <thead>
                                <tr>
                                    <th align={'left'}>
                                        {localizedStrings.card}
                                    </th>
                                    <th align={'right'} style={{textWrap: 'nowrap'}}>
                                        {localizedStrings.activated_on}
                                    </th>
                                    <th align={'left'}>
                                        {localizedStrings.recipient}
                                    </th>
                                    <th align={'right'}>
                                        {localizedStrings.balance}
                                    </th>
                                    <th align={'right'}>
                                    </th>
                                </tr>
                                </thead>
                                <tbody>
                                {giftCards.length === 0 ? (
                                        <tr>
                                            <td colSpan="5" style={{ textAlign: 'center' }}><h5>No gift cards found.</h5></td>
                                        </tr>
                                    ) : giftCards.map((giftCard, index) => (
                                    <tr key={giftCard.id}>
                                        <td align={'left'}>
                                            <HStack justify={'flex-start'}>
                                                <div className={'loopz-card-design-thumbnail-container'}>
                                                    <div className={'loopz-card-design-thumbnail-mask'}>
                                                        <img className={'loopz-card-design-thumbnail-image'}
                                                             src={giftCard.card_design_image_url}
                                                             alt={giftCard.code}/>
                                                    </div>
                                                </div>
                                                <VStack alignment={'topLeft'}>
                                                    <Text isBlock={true} style={{textWrap: 'nowrap'}}>{formatCardNumber(giftCard.code)}</Text>
                                                    <HStack alignment={'left'} justify={'flex-start'}>
                                                        <Badge intent={'default'}>
                                                            {giftCard.card_type}
                                                        </Badge>
                                                        <Badge
                                                            intent={resolveGiftCardStatusBadgeIntent(giftCard.status)}>
                                                            {giftCard.status}
                                                        </Badge>
                                                    </HStack>
                                                </VStack>
                                            </HStack>
                                        </td>
                                        <td align={'right'}>
                                            <Text>
                                                {dateFormatter.format(Date.parse(giftCard.activated_on))}
                                            </Text>
                                        </td>
                                        <td align={'left'} width={'30%'}>
                                            <VStack alignment={'topLeft'}>
                                                {giftCard.recipient_name || giftCard.recipient_emaill ? (
                                                    <>
                                                        <Text ellipsizeMode={'auto'}>{giftCard.recipient_name}</Text>
                                                        <Text ellipsizeMode={'auto'} variant={'muted'}>{giftCard.recipient_email}</Text>
                                                    </>
                                                ) : (
                                                    <Badge intent={'default'}>
                                                        {localizedStrings.anonymous}
                                                    </Badge>
                                                )}

                                            </VStack>
                                        </td>
                                        <td align={'right'}>
                                            <Text style={{textWrap: 'nowrap'}}>
                                                {giftCard.balance.formatted}
                                            </Text>
                                        </td>

                                            <td align={'right'}>
                                                    <Button
                                                        variant='tertiary'
                                                        icon={'external'}
                                                        iconPosition={'right'}
                                                        iconSize={16}
                                                        href={dashboardLoginUrl + '&redirect=' + encodeURIComponent('/gift-cards/' + giftCard.participant_id)}
                                                        target="_blank"
                                                    >
                                                        {localizedStrings.view_details}
                                                    </Button>
                                            </td>
                                    </tr>
                                    ))}
                                </tbody>
                            </table>
                        )}


                    </CardBody>
                </Card>
            </div>
        )
    }

    function renderTabPanel(tab) {
        if (tab.name === 'settings') {
            return renderSettingsTabPanel();
        } else if (tab.name === 'gift-cards') {
            return renderGiftCardsTabPanel();
        }
    }

    return (
        <>
            {notice && (
                <Notice status={notice.status} isDismissible={false}>
                    {notice.message}
                </Notice>
            )}

            {!connected ? (
                <div style={{maxWidth: '800px', margin: '2rem auto'}}>
                    <Card>
                        <CardHeader>
                            <div style={{
                                width: '100%',
                                display: 'flex',
                                flexDirection: 'column',
                                alignItems: 'center',
                                justifyContent: 'center',
                                textAlign: 'center'
                            }}>
                                <img
                                    src={window.loopzSettings?.logoUrl || '/wp-content/plugins/loopz-gift-cards/views/images/logo.png'}
                                    alt="Loopz Logo" style={{height: 80, marginBottom: 24}}/>
                                <div style={{fontSize: '1.1em', marginBottom: 24, color: '#333'}}>
                                    {localizedStrings.connect_store_description}
                                </div>
                                <Button
                                    variant="primary"
                                    onClick={handleConnect}
                                    isBusy={saving}
                                    disabled={saving}
                                >
                                    {localizedStrings.connect}
                                </Button>
                            </div>
                        </CardHeader>
                    </Card>
                </div>
                    ) : (
                    <div style={{maxWidth: '800px', margin: '2rem auto'}}>
                        <h1>{localizedStrings.loopz_gift_cards}</h1>

                        <TabPanel
                            className={'loopz-settings-tabs'}
                            initialTabName={initialTabName}
                            onSelect={() => {
                            }}
                            tabs={[
                                {
                                    name: 'gift-cards',
                                    title: localizedStrings.gift_cards
                                },
                                {
                                    name: 'settings',
                                    title: localizedStrings.settings
                                }
                            ]}
                            children={renderTabPanel}
                        />
                    </div>
                    )}
                </>
            );
            };

            const root = document.getElementById('loopz-settings-root');
if (root) {
    createRoot(root).render(<App/>);
}