<?php
/**
 * Functions to handle refunding a gift card payment on an order and to refund an order payment as a gift card funded by store credit.
 */

use Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController;

// Add a meta box for refunding a gift card payment on an order
add_action( 'add_meta_boxes', 'loopz_add_gift_card_refund_metabox' );
 // Process gift card refund via AJAX
add_action('wp_ajax_loopz_process_gift_card_refund', 'loopz_process_gift_card_refund_ajax');
// Add Refund Gift Card button to the order items table
add_action('woocommerce_order_item_add_action_buttons', 'loopz_add_refund_store_credit_button_to_order_items', 1, 1);
add_action( 'woocommerce_after_order_itemmeta', 'loopz_render_order_item_meta_admin', 10, 3 );

// Make the Loopz order line item metadata more human readable for the order admin screen
add_filter('woocommerce_order_item_get_formatted_meta_data', 'loopz_render_order_line_item_metadata', 10, 2);

function loopz_render_order_line_item_metadata($formatted_meta, $item){
    $loopz_metadata = [];

    foreach($formatted_meta as $key => $meta){
        if(strpos($meta->key, '_loopz') !== false) {
            $meta_line_item = $formatted_meta[$key];
            $loopz_metadata[$meta_line_item->key] = $meta_line_item->value;
            unset($formatted_meta[$key]);
        }
    }

    if(!empty($loopz_metadata)) {
        $recipient = null;
        $sender = null;
        $delivery = null;
        $message = null;
        $send_to_sender = false;
        if(isset($loopz_metadata['_'.LOOPZ_META_DATA_KEY_RECIPIENT_NAME])) {
            $recipient = $loopz_metadata['_'.LOOPZ_META_DATA_KEY_RECIPIENT_NAME];
        }
        if(isset($loopz_metadata['_'.LOOPZ_META_DATA_KEY_RECIPIENT_EMAIL])) {
            if(isset($recipient)) {
                $recipient .= ' - ' . $loopz_metadata['_'.LOOPZ_META_DATA_KEY_RECIPIENT_EMAIL];
            } else {
                $recipient = $loopz_metadata['_'.LOOPZ_META_DATA_KEY_RECIPIENT_EMAIL];
            }
        }
        if(isset($loopz_metadata['_'.LOOPZ_META_DATA_KEY_SENDER_NAME])) {
            $sender = $loopz_metadata['_'.LOOPZ_META_DATA_KEY_SENDER_NAME];
        }
        if(isset($loopz_metadata['_'.LOOPZ_META_DATA_KEY_SENDER_EMAIL])) {
            if(isset($sender)) {
                $sender .= ' - ' . $loopz_metadata['_'.LOOPZ_META_DATA_KEY_SENDER_EMAIL];
            } else {
                $sender = $loopz_metadata['_'.LOOPZ_META_DATA_KEY_SENDER_EMAIL];
            }
        }
        if(isset($loopz_metadata['_'.LOOPZ_META_DATA_KEY_SEND_TO_SENDER])) {
            $send_to_sender = $loopz_metadata['_'.LOOPZ_META_DATA_KEY_SEND_TO_SENDER] === 'true';
        }
        if(isset($loopz_metadata['_'.LOOPZ_META_DATA_KEY_DELIVERY])) {
            if($loopz_metadata['_'.LOOPZ_META_DATA_KEY_DELIVERY] === 'INSTANT') {
                $delivery = __('Instant', 'loopz-gift-cards');
            } else if ($loopz_metadata['_'.LOOPZ_META_DATA_KEY_DELIVERY] === 'FUTURE' && isset($loopz_metadata['_'.LOOPZ_META_DATA_KEY_DELIVERY_TIME_ISO])){
                $delivery_time = new DateTime($loopz_metadata['_'.LOOPZ_META_DATA_KEY_DELIVERY_TIME_ISO]);
                $formatted_delivery_time = wp_date(get_option('date_format') . ' ' . get_option('time_format'), $delivery_time->getTimestamp(), $delivery_time->getTimezone());
                $delivery = __('Scheduled for ', 'loopz-gift-cards').$formatted_delivery_time;
            }
        }
        if(isset($loopz_metadata['_'.LOOPZ_META_DATA_KEY_MESSAGE])) {
            $message = $loopz_metadata['_'.LOOPZ_META_DATA_KEY_MESSAGE];
        }

        if(isset($recipient) && !$send_to_sender) {
            array_push($formatted_meta, (object) array('key' => 'Recipient' , 'value' => $recipient ,'display_key' => __('Recipient', 'loopz-gift-cards'), 'display_value' => $recipient));
        }
        if(isset($sender)) {
            if($send_to_sender) {
                array_push($formatted_meta, (object) array('key' => 'Recipient' , 'value' => $sender ,'display_key' => __('Recipient', 'loopz-gift-cards'), 'display_value' => $sender));
            } else {
                array_push($formatted_meta, (object) array('key' => 'Sender' , 'value' => $sender ,'display_key' => __('Sender', 'loopz-gift-cards'), 'display_value' => $sender));
            }
        }
        if(isset($delivery)) {
            array_push($formatted_meta, (object) array('key' => 'Delivery' , 'value' => $delivery ,'display_key' => __('Delivery', 'loopz-gift-cards'), 'display_value' => $delivery));
        }
        if(isset($message)) {
            array_push($formatted_meta, (object) array('key' => 'Message' , 'value' => $message ,'display_key' => __('Message', 'loopz-gift-cards'), 'display_value' => $message));
        }
    }

    return $formatted_meta;
}

function loopz_render_order_item_meta_admin($item_id, $item, $product ) {
    // Only for line items and in admin
    if ( is_admin() && $item->is_type( 'line_item' ) ) {
        // Retrieve the custom meta data
        $custom_data = $item->get_meta( '_custom_data' );

        if ( ! empty( $custom_data ) ) {
            // Display the custom meta data
            echo '<p><strong>' . esc_html(__( 'Custom Data', 'loopz-gift-cards' )) . ':</strong> ' . esc_html( $custom_data ) . '</p>';
        }
    }
}

// Add the meta box to the order details screen
function loopz_add_gift_card_refund_metabox() {
    $hpos_enabled = false;
    if(function_exists('wc_get_container')) {
        $hpos_enabled = wc_get_container()->get(CustomOrdersTableController::class)->custom_orders_table_usage_is_enabled();
    }
    add_meta_box(
        'loopz_gift_card_refund',
        __( 'Gift Card Refund', 'loopz-gift-cards' ),
        'loopz_render_gift_card_refund_metabox',
        $hpos_enabled
            ? wc_get_page_screen_id( 'shop-order' )
            : 'shop_order',
        'side',
        'high'
    );
}

function loopz_render_gift_card_refund_metabox( $post_or_order ) {
    // Handle both legacy (post object) and HPOS (order object) cases
    if ( is_a( $post_or_order, 'WP_Post' ) ) {
        $order = wc_get_order( $post_or_order->ID );
    } else {
        $order = $post_or_order;
    }
    
    if ( ! $order || ! $order->meta_exists( LOOPZ_ORDER_META_CODE_KEY ) ) {
        echo '<p>' . esc_html__( 'No gift card applied to this order.', 'loopz-gift-cards' ) . '</p>';
        return;
    }
    
    $code = $order->get_meta( LOOPZ_ORDER_META_CODE_KEY );
    $redeemed_balance = $order->get_meta( LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY );
    
    if ( $redeemed_balance <= 0 ) {
        echo '<p>' . esc_html__( 'No gift card balance to refund.', 'loopz-gift-cards' ) . '</p>';
        return;
    }
    
    $currency_symbol = get_woocommerce_currency_symbol( $order->get_currency() );
    ?>
    <p style="margin-bottom:4px"><strong><?php esc_html(__( 'Gift card code', 'loopz-gift-cards' )); ?></strong></p>
    <p style="margin-top: 0; margin-bottom: 4px;"><?php echo esc_html( strtoupper( $code ) ); ?></p>
    <p style="margin-bottom:4px"><strong><?php esc_html(__( 'Redeemed balance', 'loopz-gift-cards' )); ?></strong></p>
    <p style="margin-top: 0; margin-bottom: 4px;"><?php echo esc_html( $currency_symbol . number_format( $redeemed_balance, 2 ) ); ?></p>
    <?php
    woocommerce_wp_text_input(
        array(
            'id'          => 'loopz_refund_amount',
            'label'       => __( 'Refund amount', 'loopz-gift-cards' ),
            'type'        => 'number',
            'step'        => '0.01',
            'min'         => '0',
            'max'         => $redeemed_balance,
            'value'       => $redeemed_balance,
            'desc_tip'    => false,
            /* translators: %1$s is currency symbol, %2$s is a formatted money amount. */
            'description' => sprintf( __( 'Maximum refundable amount: %1$s%2$s', 'loopz-gift-cards' ), $currency_symbol, number_format( $redeemed_balance, 2 ) ),
        )
    );
    ?>
    <div style="text-align: right;">
        <button type="button" class="button button-primary" id="loopz-refund-gift-card-btn" 
                data-order-id="<?php echo esc_attr( $order->get_id() ); ?>"
                data-redeemed-balance="<?php echo esc_attr( $redeemed_balance ); ?>">
            <?php echo esc_html( __('Refund gift card', 'loopz-gift-cards' )); ?>
        </button>
    </div>
    <div id="loopz-refund-gift-card-message" style="margin-top:8px;"></div>
    <script>
    jQuery(document).ready(function($) {
        $('#loopz-refund-gift-card-btn').on('click', function() {
            var button = $(this);
            var orderId = button.data('order-id');
            var redeemedBalance = parseFloat(button.data('redeemed-balance'));
            var refundAmount = parseFloat($('#loopz_refund_amount').val());
            var messageDiv = $('#loopz-refund-gift-card-message');
            
            // Clear previous messages
            messageDiv.html('');
            
            // Validation
            if (isNaN(refundAmount) || refundAmount <= 0) {
                messageDiv.html('<div class="notice notice-error"><p><?php esc_html_e( 'Please enter a valid refund amount greater than zero.', 'loopz-gift-cards' ); ?></p></div>');
                return;
            }
            
            if (refundAmount > redeemedBalance) {
                messageDiv.html('<div class="notice notice-error"><p><?php esc_html_e( 'Refund amount cannot exceed the redeemed balance.', 'loopz-gift-cards' ); ?></p></div>');
                return;
            }
            
            // Confirm refund
            if (!confirm('<?php esc_html_e( 'Are you sure you want to refund this gift card?', 'loopz-gift-cards' ); ?>')) {
                return;
            }
            
            // Disable button and show loading
            button.prop('disabled', true).text('<?php esc_html_e( 'Processing...', 'loopz-gift-cards' ); ?>');
            messageDiv.html('<div class="notice notice-info"><p><?php esc_html_e( 'Processing refund...', 'loopz-gift-cards' ); ?></p></div>');
            
            // Send AJAX request
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'loopz_process_gift_card_refund',
                    order_id: orderId,
                    refund_amount: refundAmount,
                    nonce: '<?php echo esc_attr(wp_create_nonce( 'loopz_gift_card_refund' )); ?>'
                },
                success: function(response) {
                    if (response.success) {
                        messageDiv.html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                        // Reload page after 2 seconds to show updated data
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        messageDiv.html('<div class="notice notice-error"><p>' + response.data.message + '</p></div>');
                        button.prop('disabled', false).text('<?php esc_html_e( 'Refund gift card', 'loopz-gift-cards' ); ?>');
                    }
                },
                error: function() {
                    messageDiv.html('<div class="notice notice-error"><p><?php esc_html_e( 'An error occurred while processing the refund.', 'loopz-gift-cards' ); ?></p></div>');
                    button.prop('disabled', false).text('<?php esc_html_e( 'Refund gift card', 'loopz-gift-cards' ); ?>');
                }
            });
        });
    });
    </script>
    <?php
}

function loopz_process_gift_card_refund_ajax() {
    // Verify nonce
    if (!isset($_POST['nonce'])) {
        wp_send_json_error(array('message' => __('Security check failed.', 'loopz-gift-cards')));
    }

    if(!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'loopz_gift_card_refund')) {
        wp_send_json_error(array('message' => __('Security check failed.', 'loopz-gift-cards')));
    }
    
    // Check permissions
    if (!current_user_can('edit_shop_orders')) {
        wp_send_json_error(array('message' => __('Permission denied.', 'loopz-gift-cards')));
    }

    // Check params
    if(!isset($_POST['order_id'])) {
        wp_send_json_error(array('message' => __('Order ID missing.', 'loopz-gift-cards')));
    }

    if(!isset($_POST['refund_amount'])) {
        wp_send_json_error(array('message' => __('Refund amount missing.', 'loopz-gift-cards')));
    }
    
    $order_id = intval($_POST['order_id']);
    $refund_amount = floatval($_POST['refund_amount']);
    
    // Validate order
    $order = wc_get_order($order_id);
    if (!$order) {
        wp_send_json_error(array('message' => __('Order not found.', 'loopz-gift-cards')));
    }
    
    // Check if order has gift card
    if (!$order->meta_exists(LOOPZ_ORDER_META_CODE_KEY)) {
        wp_send_json_error(array('message' => __('No gift card found on this order.', 'loopz-gift-cards')));
    }
    
    $code = $order->get_meta(LOOPZ_ORDER_META_CODE_KEY);
    $redeemed_balance = $order->get_meta(LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY);
    $charge_transaction_id = $order->get_meta('loopz_gift_card_transaction_id');
    
    // Server-side validation
    if ($refund_amount <= 0) {
        wp_send_json_error(array('message' => __('Refund amount must be greater than zero.', 'loopz-gift-cards')));
    }
    
    if ($refund_amount > $redeemed_balance) {
        wp_send_json_error(array('message' => __('Refund amount cannot exceed the redeemed balance.', 'loopz-gift-cards')));
    }
    
    if (empty($charge_transaction_id)) {
        wp_send_json_error(array('message' => __('No transaction ID found for this gift card.', 'loopz-gift-cards')));
    }
    
    // Process the refund
    $result = loopz_api::refund_gift_card($code, $charge_transaction_id, $order, $refund_amount);
    
    if ($result !== false) {
        // Update order meta
        $new_redeemed_balance = $redeemed_balance - $refund_amount;
        $order->update_meta_data(LOOPZ_ORDER_META_REDEEMED_BALANCE_KEY, $new_redeemed_balance);
        
        // Add order note
        $order->add_order_note(
            sprintf(
                /* translators: %1$s is gift card code, %2$s is a formatted money amount. */
                __('Gift card %1$s refunded: %2$s', 'loopz-gift-cards'),
                strtoupper($code),
                wc_price($refund_amount, array('currency' => $order->get_currency()))
            )
        );
        
        $order->save();
        
        wp_send_json_success(array(
            'message' => sprintf(
            /* translators: %s is a formatted money amount. */
                __('Gift card refunded successfully: %s', 'loopz-gift-cards'),
                wc_price($refund_amount, array('currency' => $order->get_currency()))
            )
        ));
    } else {
        wp_send_json_error(array('message' => __('Failed to process gift card refund. Please try again.', 'loopz-gift-cards')));
    }
}

function loopz_add_refund_store_credit_button_to_order_items($order) {
    if ( $order->meta_exists( LOOPZ_ORDER_META_CODE_KEY ) ) {
        return;
    }
    $order_id = esc_attr( $order->get_id() );
    // Calculate the remaining payment (outstanding amount to be refunded)
    $remaining_payment = $order->get_remaining_refund_amount ? $order->get_remaining_refund_amount() : (float) $order->get_total() - (float) $order->get_total_refunded();
    $remaining_payment = max(0, $remaining_payment);
    $remaining_payment_attr = esc_attr( $remaining_payment );
    // Get already refunded amount
    $already_refunded = method_exists($order, 'get_total_refunded') ? (float) $order->get_total_refunded() : 0;
    $currency_symbol = get_woocommerce_currency_symbol($order->get_currency());
    $already_refunded_formatted = $currency_symbol . number_format($already_refunded, 2);
    $remaining_payment_formatted = $currency_symbol . number_format($remaining_payment, 2);
    $customer_email = $order->get_billing_email();
    $customer_name = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();

    wp_enqueue_script(
        'loopz-refund-order-store-credit-js', // Handle
        $asset_path = plugins_url( '../build/loopz-refund-order-store-credit.js', __FILE__ ), // Path to your JS file
        array(), // Dependencies (e.g., 'jquery' if needed)
        filemtime( plugin_dir_path( __FILE__ ) . '../build/loopz-refund-order-store-credit.js' ),
        true // Load in footer
    );

    wp_set_script_translations(
        'loopz-refund-order-store-credit-js',
        'loopz-gift-cards'
    );

    $locale = get_locale();
    $customer_first_name = $order->get_billing_first_name();
    $customer_last_name = $order->get_billing_last_name();
    $customer_full_name = $customer_first_name;
    if(!empty($customer_last_name)) {
        $customer_full_name .= ' '.$customer_last_name;
    }
    $customer_email = $order->get_billing_email();

    wp_localize_script( 'loopz-refund-order-store-credit-js', 'LoopzRefundOrderStoreCreditSettings', array(
        'locale' => $locale,
        'languageTag' => substr($locale, 0, strpos($locale, '_')),
        'nonce' => wp_create_nonce( 'loopz_store_credit_refund' ),
        'customer_name' => $customer_full_name,
        'customer_email' => $customer_email,
        'localizedStrings' => array(
            'button_tool_tip' => __('Refunds the order as store credit sent to the customer\'s email address. The customer can redeem the store credit on your store for another purchase.', 'loopz-gift-cards'),
            'as_store_credit' => __('as store credit.', 'loopz-gift-cards'),
        )

    ) );
}

// AJAX handler for store credit refund
add_action('wp_ajax_loopz_process_store_credit_refund', 'loopz_process_store_credit_refund_ajax');
function loopz_process_store_credit_refund_ajax() {
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'loopz_store_credit_refund')) {
        wp_send_json_error(array('message' => __('Security check failed.', 'loopz-gift-cards')));
    }
    if (!current_user_can('edit_shop_orders')) {
        wp_send_json_error(array('message' => __('Permission denied.', 'loopz-gift-cards')));
    }

    $order_id               = isset( $_POST['order_id'] ) ? absint( $_POST['order_id'] ) : 0;
    $refund_amount          = isset( $_POST['refund_amount'] ) ? wc_format_decimal( sanitize_text_field( wp_unslash( $_POST['refund_amount'] ) ), wc_get_price_decimals() ) : 0;
    $refunded_amount        = isset( $_POST['refunded_amount'] ) ? wc_format_decimal( sanitize_text_field( wp_unslash( $_POST['refunded_amount'] ) ), wc_get_price_decimals() ) : 0;
    $refund_reason          = isset( $_POST['refund_reason'] ) ? sanitize_text_field( wp_unslash( $_POST['refund_reason'] ) ) : '';
    $line_item_qtys         = isset( $_POST['line_item_qtys'] ) ? json_decode( sanitize_text_field( wp_unslash( $_POST['line_item_qtys'] ) ), true ) : array();
    $line_item_totals       = isset( $_POST['line_item_totals'] ) ? json_decode( sanitize_text_field( wp_unslash( $_POST['line_item_totals'] ) ), true ) : array();
    $line_item_tax_totals   = isset( $_POST['line_item_tax_totals'] ) ? json_decode( sanitize_text_field( wp_unslash( $_POST['line_item_tax_totals'] ) ), true ) : array();
    $restock_refunded_items = isset( $_POST['restock_refunded_items'] ) && 'true' === $_POST['restock_refunded_items'];
    $email                  = isset( $_POST['email'] ) ? sanitize_text_field( wp_unslash( $_POST['email'] ) ) : '';
    $name                   = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';

    $order = wc_get_order($order_id);
    if (!$order) {
        wp_send_json_error(array('message' => __('Order not found.', 'loopz-gift-cards')));
    }
    if ($refund_amount <= 0) {
        wp_send_json_error(array('message' => __('Refund amount must be greater than zero.', 'loopz-gift-cards')));
    }
    if ($refund_amount > $order->get_remaining_refund_amount()) {
        wp_send_json_error(array('message' => __('Refund amount cannot exceed the order\'s available refund amount.', 'loopz-gift-cards')));
    }
    if (!is_email($email)) {
        wp_send_json_error(array('message' => __('Customer email is not valid. Unable to refund as store credit. Add a valid email to customer and retry', 'loopz-gift-cards')));
    }

    // Create the refund object.
    $line_items = array();
    $item_ids   = array_unique( array_merge( array_keys( $line_item_qtys ), array_keys( $line_item_totals ) ) );

    foreach ( $item_ids as $item_id ) {
        $line_items[ $item_id ] = array(
            'qty'          => 0,
            'refund_total' => 0,
            'refund_tax'   => array(),
        );
    }
    foreach ( $line_item_qtys as $item_id => $qty ) {
        $line_items[ $item_id ]['qty'] = max( $qty, 0 );
    }
    foreach ( $line_item_totals as $item_id => $total ) {
        $line_items[ $item_id ]['refund_total'] = wc_format_decimal( $total );
    }
    foreach ( $line_item_tax_totals as $item_id => $tax_totals ) {
        $line_items[ $item_id ]['refund_tax'] = array_filter( array_map( 'wc_format_decimal', $tax_totals ) );
    }

    $refund = wc_create_refund(array(
        'amount'         => (float) wc_format_decimal($refund_amount, wc_get_price_decimals()),
        'reason'         => __('Refunded as store credit with Loopz Gift Cards.', 'loopz-gift-cards'),
        'order_id'       => $order->get_id(),
        'line_items'     => $line_items,
        'refund_payment' => false, // Do not refund via gateway
        'restock_items'  => $restock_refunded_items,
    ));
    if (is_wp_error($refund)) {
        wp_send_json_error(array('message' => __('Failed to create store credit refund.', 'loopz-gift-cards')));
    }

    // Issue the store credit
    $store_credit_result = loopz_api::issue_store_credit($refund_amount, $name, $email, $order);
    if ($store_credit_result === false) {
        wp_send_json_error(array('message' => __('Failed to issue store credit. Please try again.', 'loopz-gift-cards')));
    }

    // Add an order note with details of the refund as store credit
    /* translators: %s is a formatted money amount. */
    $order_note = sprintf(__('Gift card payment refunded as store credit: %s', 'loopz-gift-cards'), wc_price($refund_amount, array('currency' => $order->get_currency())));
    $order_note .= ' - <a href="' . $store_credit_result['dashboard_link_url'] . '">' . __('View Details', 'loopz-gift-cards') . '</a>';
    $order->add_order_note($order_note);
    $order->save();
    /* translators: %s is a formatted money amount. */
    wp_send_json_success(array('message' => sprintf(__('Store credit refund successful: %s', 'loopz-gift-cards'), wc_price($refund_amount, array('currency' => $order->get_currency())))));
}

// Hook for adding authentication token to dashboard url links to allow to easily login to the dashboard
add_action( 'woocommerce_before_order_notes', 'loopz_modify_order_note_content', 10, 1 );
function loopz_modify_order_note_content( $checkout ) {
//    if(strpos($checkout, 'loopz.io') !== false) {
//
//    }
    return $checkout;
}
