<?php
/**
 * Loopz Gift Card Leaky Bucket
 *
 * @package  Loopz
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'Loopz_Leaky_Bucket' ) ) :

	/**
	 * Class Loopz_Gift_Card_Leaky_Bucket
	 */
	class Loopz_Leaky_Bucket {

		/**
		 * Session storage key.
		 *
		 * @var string
		 */
		const SESSION_KEY = 'loopz_gift_card_leaky_bucket';

		/**
		 * Maximum allowed attempts.
		 *
		 * @var int
		 */
		private $capacity = 10;

		/**
		 * Time window in seconds.
		 *
		 * @var int
		 */
		private $time_window = 60;

		/**
		 * Current count of attempts in the bucket.
		 *
		 * @var float
		 */
		private $attempts = 0;

		/**
		 * Timestamp of the last check (used for leaking attempts).
		 *
		 * @var int
		 */
		private $last_check = 0;

		/**
		 * Constructor.
		 *
		 * @param int $capacity .
		 * @param int $time_window.
		 */
		public function __construct( $capacity = 10, $time_window = 60 ) {
			$this->capacity    = $capacity;
			$this->time_window = $time_window;
			$this->load_from_session();
			$this->leak();

		}

		/**
		 * Check if there is available capacity to apply another gift card attempt.
		 *
		 * @return bool True if allowed, false otherwise.
		 */
		public function is_allowed() {
			return ( $this->attempts < $this->capacity );
		}

		/**
		 * Record a new attempt in the bucket.
		 *
		 * @return void
		 */
		public function increment() {

			$this->leak();
			$this->attempts++;
			$this->save_to_session();
		}

		/**
		 * Reduce the stored attempts based on how much time has passed.
		 * Implements a "leaky bucket" so the user can try again after time passes.
		 *
		 * @return void
		 */
		private function leak() {
			$now     = time();
			$elapsed = $now - $this->last_check;

			if ( $elapsed > 0 ) {
				$leak_per_second = $this->capacity / $this->time_window;
				$to_remove       = $leak_per_second * $elapsed;
				$this->attempts  = max( 0, $this->attempts - $to_remove );

				$this->last_check = $now;
				$this->save_to_session();
			}
		}

		/**
		 * Load bucket data from the WC session or initialize if not present.
		 *
		 * @return void
		 */
		private function load_from_session() {
			if ( ! function_exists( 'WC' ) || empty( WC()->session ) ) {
				return;
			}

			$data = WC()->session->get( self::SESSION_KEY );

			if ( ! empty( $data ) && is_array( $data ) ) {
				$this->attempts   = isset( $data['attempts'] ) ? floatval( $data['attempts'] ) : 0;
				$this->last_check = isset( $data['last_check'] ) ? intval( $data['last_check'] ) : time();
			} else {
				// Initialize if there's no session data.
				$this->attempts   = 0;
				$this->last_check = time();
				$this->save_to_session();
			}
		}

		/**
		 * Save the current bucket state back to the WC session.
		 *
		 * @return void
		 */
		private function save_to_session() {
			if ( function_exists( 'WC' ) && ! empty( WC()->session ) ) {
				$data = array(
					'attempts'   => $this->attempts,
					'last_check' => $this->last_check,
				);
				WC()->session->set( self::SESSION_KEY, $data );
			}
		}

		/**
		 * Get the current number of attempts.
		 *
		 * @return float
		 */
		public function get_attempts() {
			return $this->attempts;
		}

		/**
		 * Get the last check timestamp.
		 *
		 * @return int
		 */
		public function get_last_check() {
			return $this->last_check;
		}

		/**
		 * Reset the bucket if needed.
		 *
		 * @return void
		 */
		public function reset() {
			$this->attempts   = 0;
			$this->last_check = time();
			$this->save_to_session();
		}
	}

endif;
