<?php

define('AES_256_CBC', 'aes-256-cbc');

/**
 * Represents a parsed API response from the Loopz API
 */
class loopz_api_response
{
    public $success = false;
    public $code = 0;
    public $body = "";
    public $renderable_body = "";

    function __construct($response)
    {
        $body = wp_remote_retrieve_body($response);

        $this->code = wp_remote_retrieve_response_code($response);
        $this->success = $this->code >= 200 && $this->code < 300;
        $this->body = self::isJson($body) ? json_decode($body, true) : $body;
        $this->renderable_body = self::isJson($body) ? $body : '<div style="word-break: break-all; overflow-x: none; overflow-y: auto; max-height: 200px;">' . htmlentities($body) . '</div>';
    }

    function isJson($string)
    {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }
}

/**
 * Class to communicate with the Loopz API
 */
class loopz_api
{
    /**
     * Fetch the plugin's configuration from the Loopz API
     *
     * @return loopz_api_response
     */
    public static function get_config()
    {
        return self::invoke('/woocommerce/config', 'GET');
    }

    /**
     * Get a gift card remaining balance
     *
     * @return balance as decimal
     */
    public static function get_gift_card_balance($code = null)
    {
        if (empty($code)) {
            $code = loopz_cache::get_gift_card_code();
        }

        if (!empty($code)) {
            $giftcard = self::get_gift_card($code);

            if ($giftcard !== NULL
                && isset($giftcard['balance'])) {
                return self::convert_from_cents($giftcard['balance']);
            }
        }

        return 0;
    }

    public static function get_gift_card($code = null)
    {
        if (empty($code)) {
            $code = loopz_cache::get_gift_card_code();
        }

        // Return cached result if code matches
        if ($code !== null
            && loopz_cache::$giftcard !== null
            && strtolower($code) === strtolower(loopz_cache::$giftcard['code'])) {
            return loopz_cache::$giftcard;
        }

        $auth_token = loopz_api::get_auth_token();
        if (empty($auth_token)) {
            return null;
        }

        $program_id = loopz_preferences::get_program_id();
        $response = wp_remote_get(
            LOOPZ_API_BASE_URI . '/woocommerce/programs/' . urlencode($program_id) . '/gift-cards/' . urlencode($code),
            array(
                'headers' => array(
                    'X-Loopz-WooCommerce-Auth-Token' => $auth_token,
                    'X-Loopz-WooCommerce-Shop-Domain' => get_site_url(),
                    'Content-Type' => 'application/json'
                ),
                'timeout' => 30
            )
        );

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code === 200) {
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            // Map the new response data into the same structure you were using
            $gift_card = array(
                'balance' => isset($data['balance']['amount']) ? $data['balance']['amount'] : 0,
                'currency' => isset($data['balance']['currency']) ? $data['balance']['currency'] : '',
                'code' => isset($data['code']) ? $data['code'] : $code,
                'tokenId' => isset($data['id']) ? $data['id'] : '' // or however you want to store the ID
            );

            // Cache it for this request
            loopz_cache::$giftcard = $gift_card;
            return $gift_card;
        }

        return null;
    }

    public static function list_card_designs()
    {
        $auth_token = loopz_api::get_auth_token();
        if (empty($auth_token)) {
            return null;
        }

        $program_id = loopz_preferences::get_program_id();
        $response = wp_remote_get(
            LOOPZ_API_BASE_URI . '/woocommerce/programs/' . urlencode($program_id) . '/card-designs',
            array(
                'headers' => array(
                    'X-Loopz-WooCommerce-Auth-Token' => $auth_token,
                    'X-Loopz-WooCommerce-Shop-Domain' => get_site_url(),
                    'Content-Type' => 'application/json'
                ),
                'timeout' => 30
            )
        );

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code === 200) {
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            // Map the new response data into the same structure you were using
            $gift_card = array(
                'balance' => isset($data['balance']['amount']) ? $data['balance']['amount'] : 0,
                'currency' => isset($data['balance']['currency']) ? $data['balance']['currency'] : '',
                'code' => isset($data['code']) ? $data['code'] : $code,
                'tokenId' => isset($data['id']) ? $data['id'] : '' // or however you want to store the ID
            );

            // Cache it for this request
            loopz_cache::$giftcard = $gift_card;
            return $gift_card;
        }

        return null;
    }

    public static function list_gift_cards()
    {
        $program_id = loopz_preferences::get_program_id();
        return self::invoke('/woocommerce/programs/'.$program_id.'/gift-cards/search', 'GET');


//        $auth_token = loopz_api::get_auth_token();
//        if (empty($auth_token)) {
//            return null;
//        }
//
//        $program_id = loopz_preferences::get_program_id();
//        $response = wp_remote_get(
//            LOOPZ_API_BASE_URI . '/woocommerce/programs/' . urlencode($program_id) . '/gift-cards',
//            array(
//                'headers' => array(
//                    'X-Loopz-WooCommerce-Auth-Token' => $auth_token,
//        'X-Loopz-WooCommerce-Shop-Domain' => get_site_url(),
//                    'Content-Type' => 'application/json'
//                ),
//                'timeout' => 30
//            )
//        );
//
//        return $response;

//        $status_code = wp_remote_retrieve_response_code($response);
//        if ($status_code === 200) {
//            $body = wp_remote_retrieve_body($response);
//            $data = json_decode($body, true);
//
//            // Map the new response data into the same structure you were using
//            $gift_card = array(
//                'balance' => isset($data['balance']['amount']) ? $data['balance']['amount'] : 0,
//                'currency' => isset($data['balance']['currency']) ? $data['balance']['currency'] : '',
//                'code' => isset($data['code']) ? $data['code'] : $code,
//                'tokenId' => isset($data['id']) ? $data['id'] : '' // or however you want to store the ID
//            );

    }

    /**
     * Redeem a gift card
     *
     * @return integer returns the amount redeemed
     */
    public static function redeem_gift_card($code, $value, $order)
    {

        $giftcard = self::get_gift_card($code);
        if (!$giftcard) {
            return -1;
        }

        $balance = self::get_gift_card_balance($code);
        if ($balance < $value) {
            return -1;
        }

        $auth_token = self::get_auth_token();
        if (empty($auth_token)) {
            return -1;
        }

        $idempotency_key = wp_generate_uuid4();

        $payload = array(
            'idempotency_key' => $idempotency_key,
            'amount' => self::convert_to_cents($value),
            'order_id' => (string)$order->get_id(),
            'metadata' => array(
                'wcOrderId' => (string)$order->get_id(),
                'wcOrderTotal' => (string)$order->get_total()
            )
        );

        $program_id = loopz_preferences::get_program_id();
        $response = wp_remote_post(
            LOOPZ_API_BASE_URI . '/woocommerce/programs/' . urlencode($program_id) . '/gift-cards/' . urlencode($code) . '/charges',
            array(
                'headers' => array(
                    'X-Loopz-WooCommerce-Auth-Token' => $auth_token,
                    'X-Loopz-WooCommerce-Shop-Domain' => get_site_url(),
                    'Content-Type' => 'application/json'
                ),
                'body' => json_encode($payload),
                'timeout' => 45
            )
        );

        $status_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);

        if ($status_code === 200 && is_array($data)) {
            $transaction_id = !empty($data['id']) ? $data['id'] : '';

            // Return both the redeemed amount and the transaction ID
            return array(
                'redeemed_value' => $value,
                'transaction_id' => $transaction_id
            );
        }

        return -1;
    }

    /**
     * Issues a gift card with store credit
     *
     * @return integer returns the amount redeemed
     */
    public static function issue_store_credit($amount, $recipient_name, $recipient_email, $order)
    {
        $auth_token = self::get_auth_token();
        if (empty($auth_token)) {
            return -1;
        }

        $idempotency_key = wp_generate_uuid4();

        $payload = array(
            'idempotency_key' => $idempotency_key,
            'store_credit_amount' => self::convert_to_cents($amount),
            'recipient_name' => $recipient_name,
            'recipient_email' => $recipient_email,
            'order_id' => $order->get_id(),
            'payment_id' => $order->get_transaction_id(),
        );

        $program_id = loopz_preferences::get_program_id();
        $response = wp_remote_post(
            LOOPZ_API_BASE_URI . '/woocommerce/programs/' . urlencode($program_id) . '/store-credit',
            array(
                'headers' => array(
                    'X-Loopz-WooCommerce-Auth-Token' => $auth_token,
                    'X-Loopz-WooCommerce-Shop-Domain' => get_site_url(),
                    'Content-Type' => 'application/json'
                ),
                'body' => json_encode($payload),
                'timeout' => 45
            )
        );

        $status_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);

        if ($status_code === 200 && is_array($data)) {
            $gift_card_id = $data['gift_card_id'];
            $balance = $data['balance']['amount'];
            $dashboard_link_url = $data['dashboard_link_url'];

            // Return both the store credit amount and the gift card ID
            return array(
                'store_credit_amount' => $balance,
                'gift_card_id' => $gift_card_id,
                'dashboard_link_url' => $dashboard_link_url
            );
        }

        return -1;
    }

    public static function get_api_base_uri()
    {
        return LOOPZ_API_BASE_URI;
    }

    public static function dashboard_root()
    {
        if (isset($_COOKIE['loopz_dashboard_root'])) {
            return sanitize_url(wp_unslash($_COOKIE['loopz_dashboard_root']));
        }

        return 'https://dashboard.loopz.io';
    }

    /**
     * Invoke an API call to Loopz
     *
     * @return loopz_api_response
     */
    public static function invoke($endpoint, $method = "GET", $data = null, $auth_token = null)
    {
        $base_uri = self::get_api_base_uri();
        $url = esc_url_raw($base_uri . $endpoint);
        $response = false;
        $json = null;

        if ($data !== NULL) {
            $json = json_encode($data);
        }

        if ($auth_token === NULL) {
            $auth_token = self::get_auth_token();
        }

        $current_plugin_version = LOOPZ_VERSION;

        if ($current_plugin_version === NULL || strlen($current_plugin_version) <= 0) {
            $current_plugin_version = "unknown";
        }

        $args = array(
            'timeout' => 60,
            'body' => $json,
            'headers' => array(
                'X-Loopz-WooCommerce-Auth-Token' => $auth_token,
                'X-Loopz-WooCommerce-Shop-Domain' => get_site_url(),
                'content-type' => 'application/json',
                'accept' => 'application/json',
                'user-agent' => 'WordPress/Loopz-WordPress-Plugin'
            )
        );

        if ($method === "GET") {
            $response = wp_remote_get($url, $args);
        } else if ($method === "POST") {
            $response = wp_remote_post($url, $args);
        } else {
            $args = array(
                'method' => $method
            );
            $response = wp_remote_request($url, $args);
        }

        if (is_wp_error($response)) {
            $error = $response->get_error_message();

            echo '<div id="message" class="notice notice-error">';
            echo '<p>';
            echo '<strong>';
            echo 'Error talking to Loopz at ' . esc_html($url . ' - ' . $error) . '<br>';
            if (strpos($error, 'tls') !== false) {
                echo '<br>The Loopz plugin requires that your PHP version is 5.6+ and cURL supports TLS1.2.<br>';
                echo 'Please conduct a TLS 1.2 Compatibility Test via <a href="https://wordpress.org/plugins/tls-1-2-compatibility-test/" target="_blank">this plugin</a>';
            }
            echo '</strong>';
            echo '</p>';
            echo '</div>';
        }

        return new loopz_api_response($response);
    }

    private static function convert_to_cents($amount)
    {
        return $amount * pow(10, self::get_wc_currency_precision());
    }

    private static function convert_from_cents($amount)
    {
        return $amount / pow(10, self::get_wc_currency_precision());
    }

    private static function get_wc_currency_precision()
    {
        // TODO add currency check for currencies that don't use 2 decimal point precision
//        if ( !function_exists( 'get_woocommerce_currency' ) ) {
//            require_once '/includes/wc-core-functions.php';
//        }
//        $wc_currency = get_woocommerce_currency();
        return 2;
    }

    public static function get_auth_token()
    {
        global $wpdb;

        // Get every token where description includes Loopz
        $api_keys = $wpdb->get_results("
            SELECT consumer_key
            FROM {$wpdb->prefix}woocommerce_api_keys
            WHERE description LIKE 'Loopz%'
            ORDER BY key_id DESC
        ", ARRAY_A);

        if (empty($api_keys)) {
            return null;
        }

        $keys = array();
        foreach ($api_keys as $api_key) {
            array_push($keys, $api_key['consumer_key']);
        }

        return implode(',', $keys);
    }

    /**
     * Refund a gift card through the Loopz API
     */
    public static function refund_gift_card($code, $charge_transaction_id, $order, $refund_amount)
    {
        // Get a valid WooCommerce auth token
        $auth_token = self::get_auth_token();
        if (empty($auth_token)) {
            return false;
        }

        // Generate a unique idempotency key
        $idempotency_key = wp_generate_uuid4();

        $payload = array(
            'idempotency_key' => $idempotency_key,
            'charge_transaction_id' => $charge_transaction_id,
            'refund_available_amount' => !isset($refund_amount),
            'refund_amount' => $refund_amount * 100,
            'metadata' => array(
                'wcOrderId' => (string)$order->get_id(),
                'wcOrderTotal' => (string)$order->get_total(),
            ),
        );
        $program_id = loopz_preferences::get_program_id();
        $response = wp_remote_post(
            LOOPZ_API_BASE_URI . '/woocommerce/programs/' . urlencode($program_id) . '/gift-cards/' . urlencode($code) . '/charges/' . urlencode($charge_transaction_id) . '/refunds',
            array(
                'headers' => array(
                    'X-Loopz-WooCommerce-Auth-Token' => $auth_token,
                    'X-Loopz-WooCommerce-Shop-Domain' => get_site_url(),
                    'Content-Type' => 'application/json'
                ),
                'body' => json_encode($payload),
                'timeout' => 45
            )
        );

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code === 200) {
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            return $data;
        }

        return false;
    }

    /**
     * Retrieve the dashboard login link through the Loopz API
     */
    public static function get_dashboard_login_link()
    {
        // Get a valid WooCommerce auth token
        $auth_token = self::get_auth_token();
        if (empty($auth_token)) {
            return false;
        }

        $program_id = loopz_preferences::get_program_id();
        $response = wp_remote_get(
            LOOPZ_API_BASE_URI . '/woocommerce/programs/' . urlencode($program_id) . '/dashboard-link?shopDomain='. urlencode(get_site_url()) . '&token=' . urlencode($auth_token),
            array(
                'headers' => array(
                    'X-Loopz-WooCommerce-Auth-Token' => $auth_token,
                    'X-Loopz-WooCommerce-Shop-Domain' => get_site_url(),
                    'Content-Type' => 'application/json'
                ),
                'timeout' => 45
            )
        );

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code === 200) {
            $body = wp_remote_retrieve_body($response);
            return $body;
        }

        return false;
    }

    /**
     * Offboards the woo store and cancels the subscription for the store.
     */
    public static function offboard()
    {
        // Get a valid WooCommerce auth token
        $auth_token = self::get_auth_token();
        if (empty($auth_token)) {
            return false;
        }

        $program_id = loopz_preferences::get_program_id();

        $response = wp_remote_request(
            LOOPZ_API_BASE_URI . '/woocommerce/programs/' . urlencode($program_id) . '/offboarding',
            array(
                'method' => 'DELETE',
                'headers' => array(
                    'X-Loopz-WooCommerce-Auth-Token' => $auth_token,
                    'X-Loopz-WooCommerce-Shop-Domain' => get_site_url(),
                    'Content-Type' => 'application/json'
                ),
                'timeout' => 45
            )
        );

        return true;
    }

}
