<?php

namespace Loopz\Checkout_Blocks;

use Automattic\WooCommerce\Blocks\Integrations\IntegrationInterface;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for integrating with WooCommerce Blocks
 */
class Blocks_Integration implements IntegrationInterface
{

    /**
     * The name of the integration.
     *
     * @return string
     */
    public function get_name()
    {
        return 'loopz-gift-cards-blocks';
    }

    /**
     * When called invokes any initialization/setup for the integration.
     */
    public function initialize()
    {
        $this->register_scripts_and_styles();

    }


    /**
     * Registers scripts and styles for both editor and frontend.
     */
    private function register_scripts_and_styles()
    {
        // Register main integration script and style
        $this->register_main_integration();

        // Register block editor scripts
        $this->register_block_script(
            'loopz-gift-card-editor',
            'build/loopz-gift-card.js',
            'build/loopz-gift-card.asset.php'
        );

        // Register frontend scripts for blocks
        $this->register_frontend_script(
            'loopz-gift-card-frontend',
            'build/loopz-gift-card-frontend.js',
            'build/loopz-gift-card-frontend.asset.php'
        );

        // Register block styles
        $this->register_styles();
    }

    /**
     * Registers the main JS file required to add filters and Slot/Fills.
     */
    private function register_main_integration()
    {
        $script_path = 'build/index.js';
        $style_path = 'build/style-loopz-gift-card-frontend.css';

        $script_url = LOOPZ_WC_PLUGIN_DIR_URL . $script_path;
        $style_url = LOOPZ_WC_PLUGIN_DIR_URL . $style_path;

        $script_asset_path = LOOPZ_WC_PLUGIN_DIR_PATH . 'build/index.asset.php';
        $script_asset = file_exists($script_asset_path)
            ? require $script_asset_path
            : array(
                'dependencies' => array(),
                'version' => $this->get_file_version($script_path),
            );

        wp_register_style(
            'loopz-gift-card',
            $style_url,
            array(),
            $this->get_file_version($style_path)
        );

        wp_register_script(
            'loopz-gift-card-blocks',
            $script_url,
            $script_asset['dependencies'],
            $script_asset['version'],
            true
        );

        // Localize script to pass data to JavaScript
        wp_localize_script(
            'loopz-gift-card-blocks',
            'loopzGiftCardData',
            $this->get_script_data()
        );

        wp_set_script_translations(
            'loopz-gift-card-blocks',
            'loopz-gift-cards'
        );
    }

    /**
     * Registers block styles for the block editor.
     */
    private function register_styles()
    {
        $block_style_path = 'build/style-loopz-gift-card-frontend.css';
        $block_style_url = LOOPZ_WC_PLUGIN_DIR_URL . $block_style_path;

        wp_enqueue_style(
            'loopz-gift-card',
            $block_style_url,
            array(),
            $this->get_file_version($block_style_path)
        );
    }

    /**
     * Helper method to register block editor scripts.
     *
     * @param string $handle Script handle name.
     * @param string $script_path Path to the JS file.
     * @param string $asset_path Path to the asset file.
     */
    private function register_block_script($handle, $script_path, $asset_path)
    {
        $script_url = LOOPZ_WC_PLUGIN_DIR_URL . $script_path;
        $asset_file = LOOPZ_WC_PLUGIN_DIR_PATH . $asset_path;
        $asset = file_exists($asset_file) ? require $asset_file : array(
            'dependencies' => array(),
            'version' => $this->get_file_version($script_path),
        );

        wp_register_script(
            $handle,
            $script_url,
            $asset['dependencies'],
            $asset['version'],
            true
        );

        wp_set_script_translations(
            $handle,
            'loopz-gift-cards'
        );
    }

    /**
     * Helper method to register frontend scripts.
     *
     * @param string $handle Script handle name.
     * @param string $script_path Path to the JS file.
     * @param string $asset_path Path to the asset file.
     */
    private function register_frontend_script($handle, $script_path, $asset_path)
    {
        $script_url = LOOPZ_WC_PLUGIN_DIR_URL . $script_path;
        $asset_file = LOOPZ_WC_PLUGIN_DIR_PATH . $asset_path;
        $asset = file_exists($asset_file) ? require $asset_file : array(
            'dependencies' => array(),
            'version' => $this->get_file_version($script_path),
        );

        wp_register_script(
            $handle,
            $script_url,
            $asset['dependencies'],
            $asset['version'],
            true
        );

        wp_set_script_translations(
            $handle,
            'loopz-gift-cards'
        );
    }

    /**
     * Get the file modified time as a cache buster if we're in dev mode.
     *
     * @param string $file Local path to the file.
     *
     * @return string The cache buster value to use for the given file.
     */
    protected function get_file_version($file)
    {
        $file_path = LOOPZ_WC_PLUGIN_DIR_PATH . $file;
        if (defined('SCRIPT_DEBUG') && SCRIPT_DEBUG && file_exists($file_path)) {
            return filemtime($file_path);
        }

        return LOOPZ_VERSION;
    }

    /**
     * Returns an array of script handles to enqueue in the frontend context.
     *
     * @return string[]
     */
    public function get_script_handles()
    {
        return array(
            'loopz-gift-card-blocks',
            'loopz-gift-card-frontend',
        );
    }

    /**
     * Returns an array of script handles to enqueue in the editor context.
     *
     * @return string[]
     */
    public function get_editor_script_handles()
    {
        return array(
            'loopz-gift-card-blocks',
            'loopz-gift-card-editor',
        );
    }

    /**
     * Returns an array of style handles to enqueue in the frontend context.
     *
     * @return string[]
     */
    public function get_style_handles()
    {
        return array(
            'loopz-gift-card',
        );
    }

    /**
     * Returns an array of style handles to enqueue in the editor context.
     *
     * @return string[]
     */
    public function get_editor_style_handles()
    {
        return array(
            'loopz-gift-card',
        );
    }

    /**
     * An array of key, value pairs of data made available to the block on the client side.
     *
     * @return array
     */
    public function get_script_data()
    {
        $redemption_enabled = \loopz_preferences::get_redemption_enabled();
        $script_data = array(
            'pluginUrl' => LOOPZ_WC_PLUGIN_DIR_URL,
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('loopz_gift_card_nonce'),
            'redemptionEnabled' => $redemption_enabled ? 'true' : 'false',
            'localizedStrings' => array(
                'gift_card_not_found'         => __('Gift card not found.', 'loopz-gift-cards'),
                'please_enter_gift_card_code' => __('Please enter a gift card code.', 'loopz-gift-cards'),
                'applied_gift_card'           => __('Applied gift card', 'loopz-gift-cards'),
                /* translators: %s is a gift card code. */
                'gift_card_with_code'         => __('Gift card: %s', 'loopz-gift-cards'),
                /* translators: %s is a gift card code. */
                'gift_card_quoted'            => __('Gift card "%s"', 'loopz-gift-cards'),
                'add_a_gift_card'             => __('Add a gift card', 'loopz-gift-cards'),
                'enter_gift_code'             => __('Enter gift code', 'loopz-gift-cards'),
                'apply'                       => __('Apply', 'loopz-gift-cards'),
                'gift_card_code'              => __('Gift card code', 'loopz-gift-cards')
            )
        );
        return $script_data;
    }


}
