/**
 * Loopz Gift Card Button Group JavaScript
 * Handles the button group functionality for variation selection
 */

(function($) {
    'use strict';

    // Initialize button groups when document is ready
    $(document).ready(function() {
        initButtonGroups();
    });

    // Re-initialize when WooCommerce variation form is updated
    $(document.body).on('woocommerce_variation_has_changed', function() {
        initButtonGroups();
    });

    function initButtonGroups() {
        $('.loopz-variation-button-group').each(function() {
            const $buttonGroup = $(this);
            const $buttons = $buttonGroup.find('.loopz-variation-button');
            const $hiddenSelect = $buttonGroup.next('select');
            const attribute = $buttonGroup.data('attribute');

            // Add click handlers to buttons
            $buttons.on('click', function(e) {
                e.preventDefault();
                
                const $button = $(this);
                const value = $button.data('value');
                
                // Don't process if button is disabled
                if ($button.hasClass('disabled') || $button.prop('disabled')) {
                    return;
                }

                // Update button states
                $buttons.removeClass('selected');
                $button.addClass('selected');

                // Update hidden select
                if ($hiddenSelect.length) {
                    $hiddenSelect.val(value).trigger('change');
                }

                // Trigger WooCommerce variation change
                triggerVariationChange(attribute, value);
            });

            // Add keyboard navigation
            $buttons.on('keydown', function(e) {
                const $currentButton = $(this);
                const $allButtons = $buttons.filter(':not(:disabled)');
                const currentIndex = $allButtons.index($currentButton);

                switch (e.keyCode) {
                    case 37: // Left arrow
                        e.preventDefault();
                        if (currentIndex > 0) {
                            $allButtons.eq(currentIndex - 1).focus().click();
                        }
                        break;
                    case 39: // Right arrow
                        e.preventDefault();
                        if (currentIndex < $allButtons.length - 1) {
                            $allButtons.eq(currentIndex + 1).focus().click();
                        }
                        break;
                    case 13: // Enter
                    case 32: // Space
                        e.preventDefault();
                        $currentButton.click();
                        break;
                }
            });
        });
    }

    function triggerVariationChange(attribute, value) {
        // Find the variation form
        const $form = $('.variations_form');
        
        if ($form.length) {
            // Trigger WooCommerce's variation change event
            $form.trigger('woocommerce_variation_select_change');
            
            // Also trigger the found_variation event if we have a valid variation
            setTimeout(function() {
                const variationData = $form.data('product_variations');
                if (variationData && variationData.variationData) {
                    // Check if we have a valid variation
                    const variationData = variationData.variationData;
                    const selectedAttributes = {};
                    
                    // Get all selected attributes
                    $form.find('select[name^="attribute_"]').each(function() {
                        const name = $(this).attr('name').replace('attribute_', '');
                        const value = $(this).val();
                        if (value) {
                            selectedAttributes[name] = value;
                        }
                    });

                    // Find matching variation
                    let foundVariation = null;
                    for (let i = 0; i < variationData.length; i++) {
                        const variation = variationData[i];
                        let match = true;
                        
                        for (const attr in selectedAttributes) {
                            if (variation.attributes['attribute_' + attr] !== selectedAttributes[attr]) {
                                match = false;
                                break;
                            }
                        }
                        
                        if (match) {
                            foundVariation = variation;
                            break;
                        }
                    }

                    // Trigger found_variation event if we found a valid variation
                    if (foundVariation && foundVariation.is_purchasable && foundVariation.is_in_stock) {
                        $form.trigger('found_variation', [foundVariation]);
                    } else {
                        $form.trigger('reset_data');
                    }
                }
            }, 100);
        }
    }

    // Update button states when variations change
    $(document.body).on('woocommerce_update_variation_values', function(event, variation) {
        updateButtonStates(variation);
    });

    function updateButtonStates(variation) {
        $('.loopz-variation-button-group').each(function() {
            const $buttonGroup = $(this);
            const attribute = $buttonGroup.data('attribute');
            const $buttons = $buttonGroup.find('.loopz-variation-button');

            if (variation && variation.attributes) {
                const variationAttribute = 'attribute_' + attribute;
                const variationValue = variation.attributes[variationAttribute];

                // Update button states based on variation availability
                $buttons.each(function() {
                    const $button = $(this);
                    const buttonValue = $button.data('value');
                    
                    if (variationValue === buttonValue) {
                        $button.addClass('selected').removeClass('disabled');
                    } else {
                        $button.removeClass('selected');
                        
                        // Check if this combination is available
                        if (variation.is_purchasable && variation.is_in_stock) {
                            $button.removeClass('disabled');
                        } else {
                            $button.addClass('disabled');
                        }
                    }
                });
            }
        });
    }

    // Handle form reset
    $(document.body).on('woocommerce_reset_variations', function() {
        $('.loopz-variation-button').removeClass('selected disabled');
    });

    // Add loading state during AJAX requests
    $(document.body).on('woocommerce_variations_input_changed', function() {
        $('.loopz-variation-button').addClass('loading');
    });

    $(document.body).on('woocommerce_variations_input_changed_done', function() {
        $('.loopz-variation-button').removeClass('loading');
    });

})(jQuery); 